# Frontend Implementation Plan: Special Items RM & Dimension-Based RM Support

## Overview

This document outlines the frontend UI implementation plan to support both **dimension-based RM** (fabric, leather) and **special items RM** (zippers, buttons, threads) in the system. The backend API is complete and tested (100% success rate), so this plan focuses on frontend integration.

## Backend API Status

✅ **Backend Complete**: All 11 API tests passing (100% success rate)
- Product creation supports `track_by_dimensions` flag
- GRN processing handles both types correctly
- BOM validation supports mixed BOMs
- Production allocation handles both types

## Implementation Phases

### Phase 1: Product Management (Priority: High)

**Goal**: Allow users to create and edit RM products with proper tracking configuration.

#### 1.1 Update Product Creation Form (`/products/new`)
**File**: `client/app/(dashboard)/products/new/page.jsx`

**Changes Required**:
- Add `track_by_dimensions` field (checkbox) - only visible when `product_type = 'RM'`
- Add `unit_of_measure` field (dropdown: inch/cm/m) - only visible when `product_type = 'RM'` AND `track_by_dimensions = true`
- Update Zod schema to include new fields
- Add conditional logic:
  - When `product_type = 'RM'`:
    - Show "Track by Dimensions" checkbox (default: `true` for dimension-based RM)
    - If `track_by_dimensions = true`: Show "Unit of Measure" dropdown (required)
    - If `track_by_dimensions = false`: Hide unit of measure (special items RM)
  - When `product_type = 'FG'`: Hide both fields

**UI/UX Considerations**:
- Add helper text: "For fabric/leather materials, track by dimensions. For zippers/buttons, use quantity tracking."
- Show examples: "Dimension-based: Fabric, Leather | Special items: Zippers, Buttons, Threads"
- Visual distinction: Group RM-specific fields in a collapsible section or card

**Validation Rules**:
- For RM products: `track_by_dimensions` must be explicitly set (true or false)
- For dimension-based RM (`track_by_dimensions = true`): `unit_of_measure` is required
- For special items RM (`track_by_dimensions = false`): `unit_of_measure` should be null/empty

#### 1.2 Update Product Edit Form (`/products/[id]/edit`)
**File**: `client/app/(dashboard)/products/[id]/edit/page.jsx`

**Changes Required**:
- Same fields and logic as creation form
- Pre-populate `track_by_dimensions` and `unit_of_measure` from product data
- Handle editing constraints:
  - If product is dimension-based RM (`track_by_dimensions = true`), prevent disabling dimension tracking (backend enforces this)
  - Show read-only state or warning if trying to change from dimension-based to special items

**Validation Rules**:
- Same as creation form
- Additional: Prevent disabling `track_by_dimensions` if currently `true` (backend will reject)

#### 1.3 Update Product Display/Detail Page (`/products/[id]`)
**File**: `client/app/(dashboard)/products/[id]/page.jsx`

**Changes Required**:
- Display `track_by_dimensions` status (if RM product)
- Display `unit_of_measure` (if dimension-based RM)
- Add badge/indicator: "Dimension-based RM" or "Special Items RM"
- Show appropriate inventory info based on type (link to dimension-based or quantity-based inventory)

---

### Phase 2: GRN (Goods Receipt Note) Forms (Priority: High)

**Goal**: Support receiving both dimension-based and special items RM through GRN.

#### 2.1 Update GRN Creation Form (`/procurement/grns/new`)
**File**: `client/app/(dashboard)/procurement/grns/new/page.jsx`

**Changes Required**:
- Add conditional fields based on selected product's `track_by_dimensions` flag
- For **dimension-based RM** products (`track_by_dimensions = true`):
  - Show dimension fields: `piece_length`, `piece_width`, `dimension_unit`, `pieces_count`
  - Hide standard `quantity` field (or show as read-only/calculated)
  - Calculate and display area: `length × width × pieces_count`
  - Show unit of measure from product
- For **special items RM** products (`track_by_dimensions = false`):
  - Show standard `quantity` field (as currently implemented)
  - Hide dimension fields
  - No area calculation needed

**Implementation Approach**:
1. Fetch product details when product is selected in GRN item
2. Check `track_by_dimensions` flag from product
3. Dynamically show/hide fields based on flag
4. Update form validation schema conditionally

**UI/UX Considerations**:
- Use `useFieldArray` from react-hook-form (already in use)
- For each GRN item row:
  - When product selected, fetch product details
  - Show/hide fields based on product type
  - Group dimension fields in a collapsible section if needed
- Add helper text/icon to indicate which type of RM is being added
- Show calculated area for dimension-based RM items

**Validation Rules**:
- Dimension-based RM: Require `piece_length`, `piece_width`, `dimension_unit`, `pieces_count`
- Special items RM: Require `quantity` (as current)
- Mix both types in same GRN is allowed

#### 2.2 Update GRN Edit Form (if exists)
**Similar changes as creation form, but handle existing data**

#### 2.3 Update GRN Detail/View Page (`/procurement/grns/[id]`)
**File**: `client/app/(dashboard)/procurement/grns/[id]/page.jsx`

**Changes Required**:
- Display GRN items with appropriate fields:
  - Dimension-based RM: Show dimensions, pieces count, calculated area
  - Special items RM: Show quantity
- Add visual indicators/badges for each type
- Handle processing status appropriately for both types

---

### Phase 3: BOM (Bill of Materials) Forms (Priority: High)

**Goal**: Support creating BOMs with mixed RM items (dimension-based + special items).

#### 3.1 Update BOM Creation Form (`/production/boms/new`)
**File**: `client/app/(dashboard)/production/boms/new/page.jsx`

**Changes Required**:
- For each BOM item, detect RM product type (`track_by_dimensions` flag)
- For **dimension-based RM** products:
  - Show dimension requirement fields: `required_length`, `required_width`, `dimension_unit`
  - Set `use_dimensions = true` (auto-set, can be hidden)
  - Hide `quantity_per_unit` field
  - Calculate and display required area
- For **special items RM** products:
  - Show `quantity_per_unit` field
  - Set `use_dimensions = false` (auto-set, can be hidden)
  - Hide dimension requirement fields

**Implementation Approach**:
1. When RM product is selected in BOM item, fetch product details
2. Check `track_by_dimensions` flag
3. Show appropriate fields based on flag
4. Auto-set `use_dimensions` flag (true for dimension-based, false for special items)

**UI/UX Considerations**:
- Support mixed BOM items (fabric + zippers in same BOM)
- Each BOM item row should independently show fields based on its selected product
- Use visual grouping/badges to distinguish item types
- Show calculated requirements summary

**Validation Rules**:
- Dimension-based RM items: Require `required_length`, `required_width`, `dimension_unit`
- Special items RM items: Require `quantity_per_unit`
- Mixed BOMs allowed and encouraged

#### 3.2 Update BOM Edit Form (`/production/boms/[id]/edit`)
**File**: `client/app/(dashboard)/production/boms/[id]/edit/page.jsx`

**Changes Required**:
- Similar to creation form
- Pre-populate fields from existing BOM items
- Handle editing of mixed BOMs
- Prevent changing `use_dimensions` flag (tied to product type)

#### 3.3 Update BOM Detail/View Page (`/production/boms/[id]`)
**File**: `client/app/(dashboard)/production/boms/[id]/page.jsx`

**Changes Required**:
- Display BOM items with appropriate fields
- Show visual distinction between dimension-based and quantity-based items
- Display calculated requirements for each type

---

### Phase 4: Inventory Displays (Priority: Medium)

**Goal**: Show appropriate inventory information based on RM product type.

#### 4.1 Update Inventory List/Dashboard
**File**: `client/app/(dashboard)/inventory/page.jsx` (or relevant inventory pages)

**Changes Required**:
- For **dimension-based RM** products:
  - Show dimension-based inventory: total pieces, areas (full/usable/waste/scrap)
  - Link to dimension-based inventory detail page
  - Show piece count by status
- For **special items RM** products:
  - Show standard quantity-based inventory (as current)
  - Show quantity, reorder levels, etc.

**UI/UX Considerations**:
- Add filter/toggle: "Show by Type" (Dimension-based | Special Items | All)
- Use badges/indicators to distinguish types
- Different display cards/sections for each type

#### 4.2 Create/Update RM Inventory Detail Pages
**Files**: New pages or updates to existing inventory detail pages

**Changes Required**:
- Dimension-based RM detail:
  - Show piece list with dimensions
  - Filter by status (FULL, USABLE, WASTE, SCRAP)
  - Show area calculations
- Special items RM detail:
  - Show standard quantity-based inventory
  - Show movement history
  - Show reorder levels

---

### Phase 5: Production Order Interface (Priority: Medium)

**Goal**: Show production order details with appropriate material information.

#### 5.1 Update Production Order Detail/View
**File**: `client/app/(dashboard)/production/orders/[id]/page.jsx`

**Changes Required**:
- Show BOM items with appropriate display:
  - Dimension-based RM: Show dimension requirements, availability by dimensions
  - Special items RM: Show quantity requirements, availability by quantity
- Show material availability check results
- Display allocation status for both types

#### 5.2 Update Production Order Creation (if applicable)
**Changes Required**:
- Ensure compatibility with mixed BOMs
- Show appropriate validation messages

---

## Technical Implementation Details

### API Integration

**Product API**:
- Products now include `track_by_dimensions` (boolean) and `unit_of_measure` (enum: 'inch'|'cm'|'m'|null)
- When fetching products for GRN/BOM forms, include these fields

**GRN API**:
- Accepts different payloads based on product type:
  - Dimension-based RM: `piece_length`, `piece_width`, `dimension_unit`, `pieces_count`
  - Special items RM: `quantity`
- Backend routes to appropriate processing automatically

**BOM API**:
- Accepts `use_dimensions` flag per item (auto-set based on product type)
- Dimension-based items: `required_length`, `required_width`, `dimension_unit`
- Special items: `quantity_per_unit`

### Form Schema Updates

**Product Schema (Zod)**:
```typescript
const productSchema = z.object({
  // ... existing fields
  track_by_dimensions: z.boolean().optional(),
  unit_of_measure: z.enum(['inch', 'cm', 'm']).nullable().optional(),
}).refine((data) => {
  // If RM product, track_by_dimensions must be set
  if (data.product_type === 'RM') {
    return data.track_by_dimensions !== undefined;
  }
  return true;
}).refine((data) => {
  // If dimension-based RM, unit_of_measure is required
  if (data.product_type === 'RM' && data.track_by_dimensions === true) {
    return data.unit_of_measure !== null && data.unit_of_measure !== undefined;
  }
  return true;
});
```

**GRN Item Schema**:
- Conditional validation based on product `track_by_dimensions`
- Use Zod's `.refine()` or separate schemas

**BOM Item Schema**:
- Conditional validation based on product `track_by_dimensions`
- Auto-set `use_dimensions` based on product type

### Component Architecture

**Reusable Components to Create**:
1. `ProductTypeSelector` - Enhanced product type selector with RM subtype
2. `DimensionInputs` - Reusable dimension input group (length, width, unit)
3. `RMTrackingToggle` - Toggle for track_by_dimensions
4. `BOMItemForm` - Conditional BOM item form based on product type
5. `GRNItemForm` - Conditional GRN item form based on product type

### State Management

- Use React Query for product data fetching
- Use React Hook Form for form state
- Fetch product details when product is selected in GRN/BOM items
- Cache product data to avoid repeated API calls

---

## Implementation Order (Recommended)

1. **Phase 1.1**: Product Creation Form (enables creating special items RM)
2. **Phase 1.2**: Product Edit Form
3. **Phase 2.1**: GRN Creation Form (enables receiving special items RM)
4. **Phase 3.1**: BOM Creation Form (enables using special items RM in BOMs)
5. **Phase 1.3**: Product Detail Page
6. **Phase 2.3**: GRN Detail Page
7. **Phase 3.2**: BOM Edit Form
8. **Phase 3.3**: BOM Detail Page
9. **Phase 4**: Inventory Displays
10. **Phase 5**: Production Order Interface

---

## Testing Checklist

### Manual Testing
- [ ] Create dimension-based RM product (fabric)
- [ ] Create special items RM product (zipper)
- [ ] Create GRN with dimension-based RM
- [ ] Create GRN with special items RM
- [ ] Create GRN with both types (mixed)
- [ ] Create BOM with dimension-based RM only
- [ ] Create BOM with special items RM only
- [ ] Create BOM with both types (mixed)
- [ ] Edit product (change tracking type - where allowed)
- [ ] View inventory for both types
- [ ] Create production order with mixed BOM
- [ ] Confirm production with mixed materials

### Validation Testing
- [ ] Dimension-based RM requires unit_of_measure
- [ ] Special items RM doesn't require unit_of_measure
- [ ] GRN dimension fields required for dimension-based RM
- [ ] GRN quantity required for special items RM
- [ ] BOM dimension fields required for dimension-based RM
- [ ] BOM quantity_per_unit required for special items RM
- [ ] Prevent disabling track_by_dimensions for dimension-based RM (edit)

---

## UI/UX Guidelines

### Visual Indicators
- Use badges/labels: "Dimension-based" | "Special Items"
- Use icons: 📏 for dimensions, 🔢 for quantity
- Color coding: Different colors for each type (subtle)

### Help Text & Tooltips
- Explain when to use dimension-based vs special items
- Show examples for each type
- Explain dimension requirements vs quantity requirements

### Form Layout
- Group related fields together
- Use collapsible sections for optional/complex fields
- Show/hide fields smoothly (not jarring)
- Provide clear visual hierarchy

### Error Messages
- Clear validation errors for missing required fields
- Explain why fields are required/optional
- Show examples of valid input

---

## Notes

- **Backend Compatibility**: Backend already supports all required functionality
- **Backward Compatibility**: Existing RM products without `track_by_dimensions` should default to dimension-based (check backend behavior)
- **Migration**: Existing RM products may need to be updated (set `track_by_dimensions` flag)
- **Dimension-based RM UI**: This plan focuses on special items RM support. Full dimension-based RM UI (piece management, cutting operations) is a separate phase.

---

## Related Documents

- `requirements.md` - Full requirements specification
- `design.md` - System design document
- `tasks.md` - Backend implementation tasks
- `DIMENSION_BASED_INVENTORY_DESIGN.md` - Overall design document
