# Inventory Discrepancy Fix

## Problem
POS shows "Out of stock" for products that have stock according to the Inventory page. The discrepancy is:
- **Inventory Page**: Shows quantities like 300, 900, 1,185, 3,631 (In Stock)
- **POS Page**: Shows "Out of stock" badges for the same products

## Root Cause Analysis

### Issue 1: Alias Mismatch (FIXED)
- Product model uses `inventories` (plural) as association alias
- Code was using `inventory` (singular)
- **Status**: ✅ Fixed in previous commit

### Issue 2: Inventory Data Not Being Fetched/Returned Correctly
The backend code calls `stockCheckingService.getProductAvailability()` for each product, but:
1. If the service fails silently, `available_qty` might be set to 0 or null
2. Error handling might not be logging properly
3. Frontend might not be receiving the inventory data

## Solution

### Backend Fixes Applied:
1. **Always use stock checking service** - Simplified logic to always call the service (removed redundant checks)
2. **Better error handling** - Added logging with product details
3. **Fallback to included data** - If service fails, use data from Sequelize include
4. **Null safety** - Added checks for NaN and null values

### Frontend Fixes Applied:
1. **Debug logging** - Added console.log in development to see what data is received
2. **Condition check** - Already correctly checks `available_qty !== null && !== undefined`

## Testing Steps

1. **Check Backend Response**:
   ```bash
   curl "http://localhost:5000/api/products?limit=5&include=prices,inventory&quantity=1" \
     -H "Authorization: Bearer YOUR_TOKEN" | jq '.data.products[0].inventory'
   ```

2. **Check Browser Console**:
   - Open browser DevTools → Console
   - Look for "Sample product data" log
   - Verify `inventory.available_qty` has correct values

3. **Compare with Inventory Page**:
   - Check same product IDs on both pages
   - Quantities should match

## Expected Behavior After Fix

- Products with stock (e.g., 300, 900, 1,185, 3,631) should show:
  - Green badge: "300 in stock", "900 in stock", etc.
  
- Products without stock (quantity = 0) should show:
  - Red badge: "Out of stock"

- Products that don't track inventory should show:
  - No badge (null values)

## Debugging Commands

### Check Database Directly:
```sql
SELECT p.id, p.name, p.track_inventory, i.quantity, i.variant_id
FROM products p
LEFT JOIN inventories i ON p.id = i.product_id AND i.variant_id IS NULL
WHERE p.id IN (1, 2, 3, 4, 5)
ORDER BY p.id;
```

### Check Backend Logs:
```bash
tail -f server/logs/combined.log | grep -i "inventory\|available_qty"
```

## Next Steps if Issue Persists

1. **Verify stock checking service** - Check if `getProductAvailability()` is returning correct values
2. **Check product IDs match** - Ensure POS and Inventory page are looking at same products
3. **Verify variant_id handling** - Some products might have variants that need special handling
4. **Check for reserved quantities** - Reserved stock reduces available quantity

