/**
 * Response Utility
 * Standardized API response helper functions
 */

/**
 * Send a successful response
 * @param {Object} res - Express response object
 * @param {*} data - Response data
 * @param {string} message - Success message
 * @param {number} statusCode - HTTP status code (default: 200)
 */
const success = (res, data = null, message = 'Success', statusCode = 200) => {
  // Return success response with data and message
  return res.status(statusCode).json({
    success: true, // Indicate successful operation
    message, // Success message
    data, // Response data
  });
};

/**
 * Send an error response
 * @param {Object} res - Express response object
 * @param {string} message - Error message
 * @param {number} statusCode - HTTP status code (default: 500)
 * @param {*} errors - Additional error details (optional)
 */
const error = (res, message = 'Internal Server Error', statusCode = 500, errors = null) => {
  // Build error response object
  const response = {
    success: false, // Indicate failed operation
    message, // Error message
  };
  
  // Add errors object if provided
  if (errors) {
    response.errors = errors;
  }
  
  // Return error response with status code
  return res.status(statusCode).json(response);
};

/**
 * Send a validation error response
 * @param {Object} res - Express response object
 * @param {*} errors - Validation errors (usually from express-validator)
 */
const validationError = (res, errors) => {
  // Return validation error with 400 status code
  return res.status(400).json({
    success: false, // Indicate failed operation
    message: 'Validation Error', // Generic validation error message
    errors, // Validation error details
  });
};

/**
 * Send a not found response
 * @param {Object} res - Express response object
 * @param {string} message - Not found message (default: 'Resource not found')
 */
const notFound = (res, message = 'Resource not found') => {
  // Return 404 not found response
  return res.status(404).json({
    success: false, // Indicate failed operation
    message, // Not found message
  });
};

/**
 * Send an unauthorized response
 * @param {Object} res - Express response object
 * @param {string} message - Unauthorized message (default: 'Unauthorized')
 */
const unauthorized = (res, message = 'Unauthorized') => {
  // Return 401 unauthorized response
  return res.status(401).json({
    success: false, // Indicate failed operation
    message, // Unauthorized message
  });
};

/**
 * Send a forbidden response
 * @param {Object} res - Express response object
 * @param {string} message - Forbidden message (default: 'Forbidden')
 */
const forbidden = (res, message = 'Forbidden') => {
  // Return 403 forbidden response
  return res.status(403).json({
    success: false, // Indicate failed operation
    message, // Forbidden message
  });
};

// Export all response helper functions
module.exports = {
  success, // Success response helper
  error, // Error response helper
  validationError, // Validation error helper
  notFound, // Not found response helper
  unauthorized, // Unauthorized response helper
  forbidden, // Forbidden response helper
};

