/**
 * Authentication Test Helpers
 * Utilities for creating test users and getting auth tokens
 */

// Import User model
const { User } = require('../../models');
// Import JWT utility
const jwt = require('jsonwebtoken');
// Import app config for JWT secret
const config = require('../../config/app');

/**
 * Create test user
 * Creates a test user for testing purposes
 * @param {Object} userData - User data (optional)
 * @returns {Promise<Object>} Created user
 */
const createTestUser = async (userData = {}) => {
  // Default test user data
  const defaultUserData = {
    username: `testuser_${Date.now()}`, // Unique username
    password: 'password123', // Test password
    full_name: 'Test User', // Test full name
    role: 'cashier', // Default role
    email: `test_${Date.now()}@test.com`, // Unique email
    active: true, // Active user
    ...userData, // Override with provided data
  }; // Default user data
  
  // Import bcrypt for password hashing
  const bcrypt = require('bcryptjs'); // Import bcrypt
  
  // Hash password
  const hashedPassword = await bcrypt.hash(defaultUserData.password, 10); // Hash password
  
  // Create user
  const user = await User.create({
    ...defaultUserData, // Spread user data
    password: hashedPassword, // Use hashed password
  }); // Create user
  
  // Return user without password
  const userObject = user.toJSON(); // Convert to plain object
  delete userObject.password; // Remove password from object
  return userObject; // Return user object
};

/**
 * Get auth token
 * Generates JWT token for test user
 * @param {Object} user - User object
 * @returns {Promise<string>} JWT token
 */
const getAuthToken = async (user) => {
  // Generate JWT token
  const token = jwt.sign(
    {
      id: user.id, // User ID
      username: user.username, // Username
      role: user.role, // User role
    },
    config.jwtSecret, // JWT secret
    {
      expiresIn: '1h', // Token expiration (1 hour)
    }
  ); // Generate token
  
  // Return token
  return token; // Return token
};

/**
 * Create authenticated request headers
 * Creates headers with auth token for API requests
 * @param {string} token - JWT token
 * @returns {Object} Headers object
 */
const getAuthHeaders = (token) => {
  // Return headers with authorization
  return {
    Authorization: `Bearer ${token}`, // Bearer token header
  }; // Return headers
};

/**
 * Create test user with token
 * Creates a test user and returns user and token
 * @param {Object} userData - User data (optional)
 * @returns {Promise<Object>} Object with user and token
 */
const createTestUserWithToken = async (userData = {}) => {
  // Create test user
  const user = await createTestUser(userData); // Create user
  
  // Get auth token
  const token = await getAuthToken(user); // Get token
  
  // Return user and token
  return {
    user, // User object
    token, // JWT token
  }; // Return user and token
};

// Export helpers
module.exports = {
  createTestUser,
  getAuthToken,
  getAuthHeaders,
  createTestUserWithToken,
};

