/**
 * Main Routes File
 * Consolidates all module routes in a clean and organized way
 */

// Import Express Router
const express = require('express');
// Create router instance
const router = express.Router();

// Import module routes
const authRoutes = require('../modules/auth/routes');
const userRoutes = require('../modules/users/routes');
const productRoutes = require('../modules/products/routes');
const inventoryRoutes = require('../modules/inventory/routes');
const salesRoutes = require('../modules/sales/routes');
const paymentRoutes = require('../modules/payments/routes');
const categoryRoutes = require('../modules/categories/routes');
const pricingRoutes = require('../modules/pricing/routes');
const discountRoutes = require('../modules/discounts/routes');
const procurementRoutes = require('../modules/procurement/routes');
const productionRoutes = require('../modules/production/routes');
const returnRoutes = require('../modules/returns/routes');
// Customer routes removed - customer registration removed, walk-in customers only
const reportRoutes = require('../modules/reports/routes');
const labelPrintingRoutes = require('../modules/label-printing/routes');

/**
 * @swagger
 * /health:
 *   get:
 *     summary: API Health Check
 *     description: Returns API health status and server timestamp
 *     tags: [Health]
 *     responses:
 *       200:
 *         description: API is running successfully
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: true
 *                 message:
 *                   type: string
 *                   example: API is running
 *                 timestamp:
 *                   type: string
 *                   format: date-time
 *                   example: 2024-01-01T00:00:00.000Z
 */
router.get('/health', (req, res) => {
  // Return health check response
  return res.status(200).json({
    success: true, // Indicate success
    message: 'API is running', // Health message
    timestamp: new Date().toISOString(), // Current timestamp
  });
});

/**
 * @swagger
 * /version:
 *   get:
 *     summary: API Version Information
 *     description: Returns API version and name
 *     tags: [Health]
 *     responses:
 *       200:
 *         description: Version information
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: true
 *                 version:
 *                   type: string
 *                   example: 1.0.0
 *                 name:
 *                   type: string
 *                   example: Smart POS API
 */
router.get('/version', (req, res) => {
  // Return version information
  return res.status(200).json({
    success: true, // Indicate success
    version: '1.0.0', // API version
    name: 'Smart POS API', // API name
  });
});

// Mount module routes
router.use('/auth', authRoutes); // Authentication routes
router.use('/users', userRoutes); // User management routes
router.use('/products', productRoutes); // Product management routes
router.use('/inventory', inventoryRoutes);
router.use('/sales', salesRoutes);
router.use('/payments', paymentRoutes);
router.use('/categories', categoryRoutes);
router.use('/pricing', pricingRoutes);
router.use('/discounts', discountRoutes);
router.use('/procurement', procurementRoutes);
router.use('/production', productionRoutes);
router.use('/returns', returnRoutes);
// Customer routes removed - customer registration removed, walk-in customers only
router.use('/reports', reportRoutes);
router.use('/label-printing', labelPrintingRoutes); // Label printing routes

// Export router
module.exports = router;

