/**
 * Customers Routes
 * API routes for customer operations
 */

// Import Express router
const express = require('express');
// Create router instance
const router = express.Router();
// Import customer controllers
const customerControllers = require('../controllers');
// Import validation rules
const customerValidations = require('../validations');
// Import authentication middleware
const { authenticate, requireRole } = require('../../../middlewares/auth');

/**
 * @swagger
 * /api/customers:
 *   post:
 *     summary: Create customer
 *     tags: [Customers]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - name
 *             properties:
 *               name:
 *                 type: string
 *                 description: Customer name
 *               customer_type:
 *                 type: string
 *                 enum: [B2C, B2B]
 *                 description: "Customer type (default: B2C)"
 *               kra_pin:
 *                 type: string
 *                 description: KRA PIN (optional)
 *               credit_limit:
 *                 type: number
 *                 description: "Credit limit for B2B customers (default: 0)"
 *               price_list_id:
 *                 type: integer
 *                 description: Price list ID for customer-specific pricing (optional)
 */
router.post(
  '/', // POST /api/customers
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  customerValidations.validateCreateCustomer, // Validate request
  customerControllers.createCustomer // Create customer controller
);

/**
 * @swagger
 * /api/customers:
 *   get:
 *     summary: List customers
 *     tags: [Customers]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: customer_type
 *         schema:
 *           type: string
 *           enum: [B2C, B2B]
 *         description: Filter by customer type
 *       - in: query
 *         name: active
 *         schema:
 *           type: string
 *           enum: [true, false]
 *         description: "Filter by active status (default: true)"
 *       - in: query
 *         name: price_list_id
 *         schema:
 *           type: integer
 *         description: Filter by price list ID
 *       - in: query
 *         name: search
 *         schema:
 *           type: string
 *         description: Search by name or KRA PIN
 *       - in: query
 *         name: page
 *         schema:
 *           type: integer
 *         description: Page number
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *         description: Items per page
 */
router.get(
  '/', // GET /api/customers
  authenticate, // Require authentication
  customerValidations.validateListCustomers, // Validate request
  customerControllers.listCustomers // List customers controller
);

/**
 * @swagger
 * /api/customers/search:
 *   get:
 *     summary: Search customers
 *     tags: [Customers]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: q
 *         required: true
 *         schema:
 *           type: string
 *         description: Search term (searches name and KRA PIN)
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *         description: "Maximum number of results (default: 20)"
 */
router.get(
  '/search', // GET /api/customers/search
  authenticate, // Require authentication
  customerValidations.validateSearchCustomers, // Validate request
  customerControllers.searchCustomers // Search customers controller
);

/**
 * @swagger
 * /api/customers/{id}:
 *   get:
 *     summary: Get customer by ID
 *     tags: [Customers]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Customer ID
 */
router.get(
  '/:id', // GET /api/customers/:id
  authenticate, // Require authentication
  customerValidations.validateGetCustomer, // Validate request
  customerControllers.getCustomer // Get customer controller
);

/**
 * @swagger
 * /api/customers/{id}:
 *   put:
 *     summary: Update customer
 *     tags: [Customers]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Customer ID
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *                 description: Customer name
 *               customer_type:
 *                 type: string
 *                 enum: [B2C, B2B]
 *                 description: Customer type
 *               kra_pin:
 *                 type: string
 *                 description: KRA PIN (optional, can be null)
 *               credit_limit:
 *                 type: number
 *                 description: Credit limit
 *               price_list_id:
 *                 type: integer
 *                 nullable: true
 *                 description: Price list ID (optional, can be null)
 *               active:
 *                 type: boolean
 *                 description: Active status
 */
router.put(
  '/:id', // PUT /api/customers/:id
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  customerValidations.validateUpdateCustomer, // Validate request
  customerControllers.updateCustomer // Update customer controller
);

/**
 * @swagger
 * /api/customers/{id}:
 *   delete:
 *     summary: Delete customer (soft delete)
 *     tags: [Customers]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Customer ID
 */
router.delete(
  '/:id', // DELETE /api/customers/:id
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  customerValidations.validateDeleteCustomer, // Validate request
  customerControllers.deleteCustomer // Delete customer controller
);

// Export router
module.exports = router;
