/**
 * Categories Routes
 * API routes for category operations
 */

// Import Express router
const express = require('express');
// Create router instance
const router = express.Router();
// Import category controllers
const categoryControllers = require('../controllers');
// Import validation rules
const categoryValidations = require('../validations');
// Import authentication middleware
const { authenticate, requireRole, requireAnyRole } = require('../../../middlewares/auth');

/**
 * @swagger
 * /api/categories:
 *   post:
 *     summary: Create category
 *     tags: [Categories]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - name
 *             properties:
 *               name:
 *                 type: string
 *                 description: Category name
 *               parent_id:
 *                 type: integer
 *                 description: Parent category ID (optional)
 *               slug:
 *                 type: string
 *                 description: Category slug (optional, auto-generated if not provided)
 *               description:
 *                 type: string
 *                 description: Category description
 *               image_url:
 *                 type: string
 *                 description: Category image URL
 *               sort_order:
 *                 type: integer
 *                 description: Sort order for display
 *     responses:
 *       201:
 *         description: Category created successfully
 *       400:
 *         description: Validation error
 */
router.post(
  '/', // POST /api/categories
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  categoryValidations.validateCreateCategory, // Validate request
  categoryControllers.createCategory // Create category controller
);

/**
 * @swagger
 * /api/categories:
 *   get:
 *     summary: List categories
 *     tags: [Categories]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: page
 *         schema:
 *           type: integer
 *           default: 1
 *         description: Page number
 *       - in: query
 *         name: limit
 *         schema:
 *           type: integer
 *           default: 10
 *         description: Items per page
 *       - in: query
 *         name: parent_id
 *         schema:
 *           type: integer
 *           nullable: true
 *         description: Filter by parent ID (use 'null' for root categories)
 *       - in: query
 *         name: with_children
 *         schema:
 *           type: boolean
 *         description: Include child categories
 *     responses:
 *       200:
 *         description: Categories retrieved successfully
 */
router.get(
  '/', // GET /api/categories
  authenticate, // Require authentication
  categoryValidations.validateListCategories, // Validate request
  categoryControllers.listCategories // List categories controller
);

/**
 * @swagger
 * /api/categories/hierarchy:
 *   get:
 *     summary: Get category hierarchy (tree structure)
 *     tags: [Categories]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: root_category_id
 *         schema:
 *           type: integer
 *         description: Root category ID (optional, returns all root categories if not provided)
 *     responses:
 *       200:
 *         description: Category hierarchy retrieved successfully
 */
router.get(
  '/hierarchy', // GET /api/categories/hierarchy
  authenticate, // Require authentication
  categoryValidations.validateGetCategoryHierarchy, // Validate request
  categoryControllers.getCategoryHierarchy // Get category hierarchy controller
);

/**
 * @swagger
 * /api/categories/{id}:
 *   get:
 *     summary: Get category by ID
 *     tags: [Categories]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Category ID
 *     responses:
 *       200:
 *         description: Category retrieved successfully
 *       404:
 *         description: Category not found
 */
router.get(
  '/:id', // GET /api/categories/:id
  authenticate, // Require authentication
  categoryValidations.validateGetCategory, // Validate request
  categoryControllers.getCategory // Get category controller
);

/**
 * @swagger
 * /api/categories/{id}:
 *   put:
 *     summary: Update category
 *     tags: [Categories]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Category ID
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *               parent_id:
 *                 type: integer
 *                 nullable: true
 *               slug:
 *                 type: string
 *               description:
 *                 type: string
 *               image_url:
 *                 type: string
 *               sort_order:
 *                 type: integer
 *     responses:
 *       200:
 *         description: Category updated successfully
 *       400:
 *         description: Validation error
 *       404:
 *         description: Category not found
 */
router.put(
  '/:id', // PUT /api/categories/:id
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  categoryValidations.validateUpdateCategory, // Validate request
  categoryControllers.updateCategory // Update category controller
);

/**
 * @swagger
 * /api/categories/{id}:
 *   delete:
 *     summary: Delete category
 *     tags: [Categories]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: Category ID
 *     responses:
 *       200:
 *         description: Category deleted successfully
 *       400:
 *         description: Validation error (category has children or products)
 *       404:
 *         description: Category not found
 */
router.delete(
  '/:id', // DELETE /api/categories/:id
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  categoryValidations.validateDeleteCategory, // Validate request
  categoryControllers.deleteCategory // Delete category controller
);

/**
 * @swagger
 * /api/categories/products/{productId}:
 *   get:
 *     summary: Get product categories
 *     tags: [Categories]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: productId
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product ID
 *     responses:
 *       200:
 *         description: Product categories retrieved successfully
 */
router.get(
  '/products/:productId', // GET /api/categories/products/:productId
  authenticate, // Require authentication
  categoryValidations.validateGetProductCategories, // Validate request
  categoryControllers.getProductCategories // Get product categories controller
);

/**
 * @swagger
 * /api/categories/products/{productId}:
 *   post:
 *     summary: Assign product to categories
 *     tags: [Categories]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: productId
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product ID
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - category_ids
 *             properties:
 *               category_ids:
 *                 type: array
 *                 items:
 *                   type: integer
 *                 description: Array of category IDs
 *               primary_category_id:
 *                 type: integer
 *                 description: Primary category ID (optional)
 *     responses:
 *       201:
 *         description: Product assigned to categories successfully
 *       400:
 *         description: Validation error
 *       404:
 *         description: Product or category not found
 */
router.post(
  '/products/:productId', // POST /api/categories/products/:productId
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  categoryValidations.validateAssignProductToCategories, // Validate request
  categoryControllers.assignProductToCategories // Assign product to categories controller
);

/**
 * @swagger
 * /api/categories/products/{productId}/{categoryId}:
 *   delete:
 *     summary: Remove product from category
 *     tags: [Categories]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: productId
 *         required: true
 *         schema:
 *           type: integer
 *         description: Product ID
 *       - in: path
 *         name: categoryId
 *         required: true
 *         schema:
 *           type: integer
 *         description: Category ID
 *     responses:
 *       200:
 *         description: Product removed from category successfully
 *       404:
 *         description: Product-category assignment not found
 */
router.delete(
  '/products/:productId/:categoryId', // DELETE /api/categories/products/:productId/:categoryId
  authenticate, // Require authentication
  requireRole(['manager', 'system_admin']), // Require role
  categoryValidations.validateRemoveProductFromCategory, // Validate request
  categoryControllers.removeProductFromCategory // Remove product from category controller
);

// Export router
module.exports = router;

