/**
 * Authentication Validation Rules
 * Input validation rules for authentication endpoints
 */

// Import express-validator for request validation
const { body, validationResult } = require('express-validator');
// Import custom validation error class
const { ValidationError } = require('../../../utils/errors');

/**
 * Validation middleware wrapper
 * Checks validation results and returns errors if validation fails
 * @param {Object} req - Express request object
 * @param {Object} res - Express response object
 * @param {Function} next - Express next middleware function
 */
const validate = (req, res, next) => {
  // Get validation errors from request
  const errors = validationResult(req);
  
  // If validation errors exist, return error response
  if (!errors.isEmpty()) {
    // Format validation errors
    const formattedErrors = errors.array().map((error) => ({
      field: error.path || error.param, // Field name
      message: error.msg, // Error message
      value: error.value, // Invalid value
    }));
    
    // Throw validation error with formatted errors
    throw new ValidationError('Validation failed', formattedErrors);
  }
  
  // If validation passes, proceed to next middleware
  next();
};

/**
 * Login validation rules
 * Validates username and password for login endpoint
 */
const validateLogin = [
  // Validate username field
  body('username')
    .trim() // Remove whitespace
    .notEmpty() // Username is required
    .withMessage('Username is required') // Error message
    .isLength({ min: 3, max: 100 }) // Username length validation
    .withMessage('Username must be between 3 and 100 characters'), // Error message
  
  // Validate password field
  body('password')
    .notEmpty() // Password is required
    .withMessage('Password is required') // Error message
    .isLength({ min: 6 }) // Minimum password length
    .withMessage('Password must be at least 6 characters'), // Error message
  
  // Run validation middleware
  validate,
];

/**
 * PIN login validation rules
 * Validates username and PIN code for quick POS login
 */
const validateLoginWithPin = [
  // Validate username field
  body('username')
    .trim() // Remove whitespace
    .notEmpty() // Username is required
    .withMessage('Username is required') // Error message
    .isLength({ min: 3, max: 100 }) // Username length validation
    .withMessage('Username must be between 3 and 100 characters'), // Error message
  
  // Validate PIN code field
  body('pin_code')
    .notEmpty() // PIN code is required
    .withMessage('PIN code is required') // Error message
    .isLength({ min: 4, max: 6 }) // PIN code length (4-6 digits)
    .withMessage('PIN code must be between 4 and 6 characters') // Error message
    .isNumeric() // PIN code must be numeric
    .withMessage('PIN code must be numeric'), // Error message
  
  // Run validation middleware
  validate,
];

// Export validation rules
module.exports = {
  validateLogin, // Login validation rules
  validateLoginWithPin, // PIN login validation rules
};

