/**
 * Authentication Routes
 * Route definitions for authentication endpoints
 */

// Import Express Router
const express = require('express');
// Create router instance
const router = express.Router();
// Import authentication controllers
const authControllers = require('../controllers');
// Import authentication validations
const { validateLogin, validateLoginWithPin } = require('../validations');
const { validateSetup } = require('../validations/setup');
// Import authentication middleware
const { authenticate } = require('../../../middlewares/auth');

/**
 * @swagger
 * /auth/login:
 *   post:
 *     summary: User login with username and password
 *     description: Authenticates user with username and password, returns JWT token
 *     tags: [Authentication]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - username
 *               - password
 *             properties:
 *               username:
 *                 type: string
 *                 example: admin
 *               password:
 *                 type: string
 *                 format: password
 *                 example: admin123
 *     responses:
 *       200:
 *         description: Login successful
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: true
 *                 message:
 *                   type: string
 *                   example: Login successful
 *                 data:
 *                   type: object
 *                   properties:
 *                     user:
 *                       type: object
 *                     token:
 *                       type: string
 *       401:
 *         description: Invalid credentials
 */
router.post('/login', validateLogin, authControllers.login);

/**
 * @swagger
 * /auth/login-pin:
 *   post:
 *     summary: Quick POS login with username and PIN
 *     description: Authenticates user with username and PIN code for quick POS access
 *     tags: [Authentication]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - username
 *               - pin_code
 *             properties:
 *               username:
 *                 type: string
 *                 example: cashier
 *               pin_code:
 *                 type: string
 *                 example: "9999"
 *     responses:
 *       200:
 *         description: PIN login successful
 *       401:
 *         description: Invalid credentials or PIN not configured
 */
router.post('/login-pin', validateLoginWithPin, authControllers.loginWithPin);

/**
 * @swagger
 * /auth/profile:
 *   get:
 *     summary: Get current user profile
 *     description: Returns the authenticated user's profile information
 *     tags: [Authentication]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: Profile retrieved successfully
 *       401:
 *         description: Unauthorized - Invalid or missing token
 */
router.get('/profile', authenticate, authControllers.getProfile);

/**
 * @swagger
 * /auth/setup:
 *   post:
 *     summary: Initialize system and create first admin user
 *     description: Creates the first admin user when database is empty. This endpoint only works when no users exist.
 *     tags: [Authentication]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - username
 *               - email
 *               - password
 *               - full_name
 *             properties:
 *               username:
 *                 type: string
 *                 example: admin
 *               email:
 *                 type: string
 *                 format: email
 *                 example: admin@xyzpos.com
 *               password:
 *                 type: string
 *                 format: password
 *                 example: admin123
 *               full_name:
 *                 type: string
 *                 example: System Administrator
 *               pin_code:
 *                 type: string
 *                 example: "1234"
 *     responses:
 *       201:
 *         description: System initialized successfully
 *       403:
 *         description: System already initialized
 *       400:
 *         description: Validation error
 */
router.post('/setup', validateSetup, authControllers.setup);

/**
 * @swagger
 * /auth/refresh:
 *   post:
 *     summary: Refresh JWT token
 *     description: Refreshes JWT token using refresh token (not yet implemented)
 *     tags: [Authentication]
 *     responses:
 *       501:
 *         description: Not implemented
 */
router.post('/refresh', authControllers.refreshToken);

// Export router
module.exports = router;

