/**
 * System Log Model
 * Represents system audit logs (tracks user actions and system events)
 */

module.exports = (sequelize, DataTypes) => {
  // Define SystemLog model
  const SystemLog = sequelize.define('SystemLog', {
    // Primary key
    id: {
      type: DataTypes.BIGINT, // Use BIGINT for large scale
      primaryKey: true, // Set as primary key
      autoIncrement: true, // Auto-increment ID
    },
    // Foreign key to User (optional - null for system actions)
    user_id: {
      type: DataTypes.BIGINT, // Match User ID type
      allowNull: true, // User ID is optional (for system-generated logs)
      references: {
        model: 'users', // Reference to users table
        key: 'id', // Reference to id column
      },
      onUpdate: 'CASCADE', // Cascade update on user update
      onDelete: 'SET NULL', // Set to null when user is deleted
    },
    // Action performed (e.g., 'CREATE_SALE', 'UPDATE_INVENTORY')
    action: {
      type: DataTypes.STRING(100), // Action name with max length
      allowNull: false, // Action is required
      validate: {
        notEmpty: true, // Action cannot be empty string
        len: [1, 100], // Length validation
      },
    },
    // Entity type (e.g., 'Sale', 'Product', 'Inventory')
    entity: {
      type: DataTypes.STRING(100), // Entity name with max length
      allowNull: false, // Entity is required
      validate: {
        notEmpty: true, // Entity cannot be empty string
        len: [1, 100], // Length validation
      },
    },
    // Entity ID (ID of the affected entity)
    entity_id: {
      type: DataTypes.BIGINT, // Entity ID
      allowNull: true, // Entity ID is optional (for actions without specific entity)
    },
    // Additional payload data stored as JSON
    payload: {
      type: DataTypes.JSON, // JSON field for additional log data
      allowNull: true, // Payload is optional
    },
  }, {
    // Model options
    tableName: 'system_logs', // Explicit table name
    underscored: true, // Use snake_case for database columns
    timestamps: true, // Enable createdAt and updatedAt timestamps
    createdAt: 'created_at', // Map createdAt to created_at column
    updatedAt: false, // No updatedAt for logs (immutable records)
    indexes: [
      // Index on user_id for faster lookups
      {
        fields: ['user_id'], // Index on user_id field
      },
      // Index on action for filtering
      {
        fields: ['action'], // Index on action field
      },
      // Index on entity for filtering
      {
        fields: ['entity'], // Index on entity field
      },
      // Index on entity_id for lookups
      {
        fields: ['entity_id'], // Index on entity_id field
      },
      // Composite index on entity and entity_id for common queries
      {
        fields: ['entity', 'entity_id'], // Composite index
      },
      // Index on created_at for date-based queries
      {
        fields: ['created_at'], // Index on created_at field
      },
    ],
  });

  // Define model associations (if User model exists)
  SystemLog.associate = (models) => {
    // SystemLog belongs to User (many-to-one relationship, optional)
    // Only if User model exists
    if (models.User) {
      SystemLog.belongsTo(models.User, {
        foreignKey: 'user_id', // Foreign key in SystemLog table
        as: 'user', // Alias for association
      });
    }
  };

  // Return SystemLog model
  return SystemLog;
};

