/**
 * Purchase Order Item Model
 * Represents items in a purchase order
 */

module.exports = (sequelize, DataTypes) => {
  // Define PurchaseOrderItem model
  const PurchaseOrderItem = sequelize.define('PurchaseOrderItem', {
    // Primary key
    id: {
      type: DataTypes.BIGINT, // Use BIGINT for large scale
      primaryKey: true, // Set as primary key
      autoIncrement: true, // Auto-increment ID
    },
    // Foreign key to PurchaseOrder
    purchase_order_id: {
      type: DataTypes.BIGINT, // Match PurchaseOrder ID type
      allowNull: false, // Purchase order ID is required
      references: {
        model: 'purchase_orders', // Reference to purchase_orders table
        key: 'id', // Reference to id column
      },
      onUpdate: 'CASCADE', // Cascade update on PO update
      onDelete: 'CASCADE', // Cascade delete when PO is deleted
    },
    // Foreign key to Product
    product_id: {
      type: DataTypes.BIGINT, // Match Product ID type
      allowNull: false, // Product ID is required
      references: {
        model: 'products', // Reference to products table
        key: 'id', // Reference to id column
      },
      onUpdate: 'CASCADE', // Cascade update on product update
    },
    // Quantity ordered
    quantity: {
      type: DataTypes.DECIMAL(12, 3), // Decimal with 12 digits, 3 decimal places
      allowNull: false, // Quantity is required
      validate: {
        min: 0, // Quantity cannot be negative
      },
    },
    // Unit cost per item
    unit_cost: {
      type: DataTypes.DECIMAL(12, 2), // Decimal with 12 digits, 2 decimal places
      allowNull: false, // Unit cost is required
      validate: {
        min: 0, // Unit cost cannot be negative
      },
    },
    // Dimension fields for RM products (similar to GRNItem)
    piece_length: {
      type: DataTypes.DECIMAL(12, 3), // Length of each piece ordered
      allowNull: true, // Optional - only for RM products with dimensions
      validate: {
        min: 0, // Length cannot be negative
      },
    },
    piece_width: {
      type: DataTypes.DECIMAL(12, 3), // Width of each piece ordered
      allowNull: true, // Optional - only for RM products with dimensions
      validate: {
        min: 0, // Width cannot be negative
      },
    },
    dimension_unit: {
      type: DataTypes.ENUM('inch', 'cm', 'm'), // Unit of measure for piece dimensions
      allowNull: true, // Optional - only for RM products with dimensions
    },
    pieces_count: {
      type: DataTypes.INTEGER, // Number of pieces with these dimensions
      allowNull: true, // Optional - defaults to quantity if not specified
      defaultValue: null, // No default - will be set based on quantity if not provided
      validate: {
        min: 1, // Must have at least 1 piece if specified
      },
    },
  }, {
    // Model options
    tableName: 'purchase_order_items', // Explicit table name
    underscored: true, // Use snake_case for database columns
    timestamps: false, // No timestamps for PO items
    indexes: [
      // Index on purchase_order_id for faster lookups
      {
        fields: ['purchase_order_id'], // Index on purchase_order_id field
      },
      // Index on product_id for faster lookups
      {
        fields: ['product_id'], // Index on product_id field
      },
    ],
  });

  // Define model associations
  PurchaseOrderItem.associate = (models) => {
    // PurchaseOrderItem belongs to PurchaseOrder (many-to-one relationship)
    PurchaseOrderItem.belongsTo(models.PurchaseOrder, {
      foreignKey: 'purchase_order_id', // Foreign key in PurchaseOrderItem table
      as: 'purchaseOrder', // Alias for association
    });
    
    // PurchaseOrderItem belongs to Product (many-to-one relationship)
    PurchaseOrderItem.belongsTo(models.Product, {
      foreignKey: 'product_id', // Foreign key in PurchaseOrderItem table
      as: 'product', // Alias for association
    });
    
  };

  // Return PurchaseOrderItem model
  return PurchaseOrderItem;
};

