/**
 * GRN (Goods Receipt Note) Model
 * Represents goods received (can be with or without PO - open market friendly)
 */

module.exports = (sequelize, DataTypes) => {
  // Define GRN model
  const GRN = sequelize.define('GRN', {
    // Primary key
    id: {
      type: DataTypes.BIGINT, // Use BIGINT for large scale
      primaryKey: true, // Set as primary key
      autoIncrement: true, // Auto-increment ID
    },
    // Foreign key to PurchaseOrder (optional - null for open-market GRNs)
    purchase_order_id: {
      type: DataTypes.BIGINT, // Match PurchaseOrder ID type
      allowNull: true, // Purchase order ID is optional (allows GRN without PO)
      references: {
        model: 'purchase_orders', // Reference to purchase_orders table
        key: 'id', // Reference to id column
      },
      onUpdate: 'CASCADE', // Cascade update on PO update
    },
    // Foreign key to Vendor (added enhancement for better tracking)
    vendor_id: {
      type: DataTypes.BIGINT, // Match Vendor ID type
      allowNull: true, // Vendor ID is optional (for open-market)
      references: {
        model: 'vendors', // Reference to vendors table
        key: 'id', // Reference to id column
      },
      onUpdate: 'CASCADE', // Cascade update on vendor update
    },
    // Date and time when goods were received
    received_at: {
      type: DataTypes.DATE, // Date and time
      allowNull: false, // Received date is required
      defaultValue: DataTypes.NOW, // Default to current date/time
    },
    // Date and time when GRN was processed (inventory updated)
    processed_at: {
      type: DataTypes.DATE, // Date and time
      allowNull: true, // Processed date is optional (only set when processed)
    },
  }, {
    // Model options
    tableName: 'grns', // Explicit table name
    underscored: true, // Use snake_case for database columns
    timestamps: true, // Enable createdAt and updatedAt timestamps
    createdAt: 'created_at', // Map createdAt to created_at column
    updatedAt: 'updated_at', // Map updatedAt to updated_at column
    indexes: [
      // Index on purchase_order_id for faster lookups
      {
        fields: ['purchase_order_id'], // Index on purchase_order_id field
      },
      // Index on vendor_id for faster lookups
      {
        fields: ['vendor_id'], // Index on vendor_id field
      },
      // Index on received_at for date-based queries
      {
        fields: ['received_at'], // Index on received_at field
      },
    ],
  });

  // Define model associations
  GRN.associate = (models) => {
    // GRN belongs to PurchaseOrder (many-to-one relationship, optional)
    GRN.belongsTo(models.PurchaseOrder, {
      foreignKey: 'purchase_order_id', // Foreign key in GRN table
      as: 'purchaseOrder', // Alias for association
    });
    
    // GRN belongs to Vendor (many-to-one relationship, optional)
    GRN.belongsTo(models.Vendor, {
      foreignKey: 'vendor_id', // Foreign key in GRN table
      as: 'vendor', // Alias for association
    });
    
    // GRN has many GRNItems (one-to-many relationship)
    GRN.hasMany(models.GRNItem, {
      foreignKey: 'grn_id', // Foreign key in GRNItems table
      as: 'items', // Alias for association
      onDelete: 'CASCADE', // Cascade delete items when GRN is deleted
    });
  };

  // Return GRN model
  return GRN;
};

