/**
 * Discount Rule Model
 * Defines which products/categories/customers the discount applies to
 */

module.exports = (sequelize, DataTypes) => {
  // Define DiscountRule model
  const DiscountRule = sequelize.define('DiscountRule', {
    // Primary key
    id: {
      type: DataTypes.BIGINT, // Use BIGINT for large scale
      primaryKey: true, // Set as primary key
      autoIncrement: true, // Auto-increment ID
    },
    // Foreign key to Discount
    discount_id: {
      type: DataTypes.BIGINT, // Match Discount ID type
      allowNull: false, // Discount ID is required
      references: {
        model: 'discounts', // Reference to discounts table
        key: 'id', // Reference to id column
      },
      onUpdate: 'CASCADE', // Cascade update on discount update
      onDelete: 'CASCADE', // Cascade delete when discount is deleted
    },
    // Rule type (PRODUCT, CATEGORY, CUSTOMER, ALL)
    rule_type: {
      type: DataTypes.ENUM('PRODUCT', 'CATEGORY', 'CUSTOMER', 'ALL'), // Enum for rule type
      allowNull: false, // Rule type is required
      validate: {
        isIn: [['PRODUCT', 'CATEGORY', 'CUSTOMER', 'ALL']], // Validate enum values
      },
    },
    // Foreign key to Product (if rule_type is PRODUCT)
    product_id: {
      type: DataTypes.BIGINT, // Match Product ID type
      allowNull: true, // Product ID is optional (only if rule_type is PRODUCT)
      references: {
        model: 'products', // Reference to products table
        key: 'id', // Reference to id column
      },
      onUpdate: 'CASCADE', // Cascade update on product update
    },
    // Foreign key to Category (if rule_type is CATEGORY)
    category_id: {
      type: DataTypes.BIGINT, // Match Category ID type
      allowNull: true, // Category ID is optional (only if rule_type is CATEGORY)
      references: {
        model: 'categories', // Reference to categories table
        key: 'id', // Reference to id column
      },
      onUpdate: 'CASCADE', // Cascade update on category update
    },
    // Foreign key to Customer (if rule_type is CUSTOMER)
    customer_id: {
      type: DataTypes.BIGINT, // Match Customer ID type
      allowNull: true, // Customer ID is optional (only if rule_type is CUSTOMER)
      references: {
        model: 'customers', // Reference to customers table
        key: 'id', // Reference to id column
      },
      onUpdate: 'CASCADE', // Cascade update on customer update
    },
    // Minimum quantity required for discount (for quantity-based discounts)
    min_quantity: {
      type: DataTypes.DECIMAL(12, 3), // Decimal with 12 digits, 3 decimal places
      defaultValue: 1, // Default minimum quantity is 1
      allowNull: false, // Minimum quantity is required
      validate: {
        min: 0, // Minimum quantity cannot be negative
      },
    },
  }, {
    // Model options
    tableName: 'discount_rules', // Explicit table name
    underscored: true, // Use snake_case for database columns
    timestamps: false, // No timestamps for discount rules
    indexes: [
      // Index on discount_id for faster lookups
      {
        fields: ['discount_id'], // Index on discount_id field
      },
      // Index on rule_type for filtering
      {
        fields: ['rule_type'], // Index on rule_type field
      },
      // Index on product_id for faster lookups
      {
        fields: ['product_id'], // Index on product_id field
      },
      // Index on category_id for faster lookups
      {
        fields: ['category_id'], // Index on category_id field
      },
      // Index on customer_id for faster lookups
      {
        fields: ['customer_id'], // Index on customer_id field
      },
    ],
  });

  // Define model associations
  DiscountRule.associate = (models) => {
    // DiscountRule belongs to Discount (many-to-one relationship)
    DiscountRule.belongsTo(models.Discount, {
      foreignKey: 'discount_id', // Foreign key in DiscountRule table
      as: 'discount', // Alias for association
    });
    
    // DiscountRule belongs to Product (many-to-one relationship, optional)
    DiscountRule.belongsTo(models.Product, {
      foreignKey: 'product_id', // Foreign key in DiscountRule table
      as: 'product', // Alias for association
    });
    
    // DiscountRule belongs to Category (many-to-one relationship, optional)
    DiscountRule.belongsTo(models.Category, {
      foreignKey: 'category_id', // Foreign key in DiscountRule table
      as: 'category', // Alias for association
    });
    
    // DiscountRule belongs to Customer (many-to-one relationship, optional)
    DiscountRule.belongsTo(models.Customer, {
      foreignKey: 'customer_id', // Foreign key in DiscountRule table
      as: 'customer', // Alias for association
    });
  };

  // Return DiscountRule model
  return DiscountRule;
};

