/**
 * Customer Model
 * Represents customers (B2B, B2C) - Optional for walk-in sales
 */

module.exports = (sequelize, DataTypes) => {
  // Define Customer model
  const Customer = sequelize.define('Customer', {
    // Primary key
    id: {
      type: DataTypes.BIGINT, // Use BIGINT for large scale
      primaryKey: true, // Set as primary key
      autoIncrement: true, // Auto-increment ID
    },
    // Customer name
    name: {
      type: DataTypes.STRING(150), // Customer name with max length
      allowNull: false, // Name is required
      validate: {
        notEmpty: true, // Name cannot be empty string
      },
    },
    // Customer type (B2C or B2B)
    customer_type: {
      type: DataTypes.ENUM('B2C', 'B2B'), // Enum for customer type
      defaultValue: 'B2C', // Default to B2C (walk-in customers)
      allowNull: false, // Type is required
    },
    // Email address (optional)
    email: {
      type: DataTypes.STRING(255), // Email address with max length
      allowNull: true, // Email is optional
      validate: {
        isEmail: true, // Validate email format
        len: [0, 255], // Maximum length validation
      },
    },
    // Phone number (optional)
    phone: {
      type: DataTypes.STRING(20), // Phone number with max length
      allowNull: true, // Phone is optional
      validate: {
        len: [0, 20], // Maximum length validation
      },
    },
    // KRA PIN (optional - may not be registered)
    kra_pin: {
      type: DataTypes.STRING(20), // KRA PIN format
      allowNull: true, // PIN is optional
      validate: {
        len: [0, 20], // Maximum length validation
      },
    },
    // Credit limit for B2B customers
    credit_limit: {
      type: DataTypes.DECIMAL(12, 2), // Decimal with 12 digits, 2 decimal places
      defaultValue: 0, // Default credit limit is 0
      allowNull: false, // Credit limit is required
      validate: {
        min: 0, // Credit limit cannot be negative
      },
    },
    // Active status (soft delete)
    active: {
      type: DataTypes.BOOLEAN, // Boolean for active status
      defaultValue: true, // Default to active
      allowNull: false, // Active status is required
    },
    // Foreign key to PriceList (for customer-specific pricing)
    price_list_id: {
      type: DataTypes.BIGINT, // Match PriceList ID type
      allowNull: true, // Price list ID is optional
      references: {
        model: 'price_lists', // Reference to price_lists table
        key: 'id', // Reference to id column
      },
      onUpdate: 'CASCADE', // Cascade update on price list update
    },
  }, {
    // Model options
    tableName: 'customers', // Explicit table name
    underscored: true, // Use snake_case for database columns
    timestamps: true, // Enable createdAt and updatedAt timestamps
    createdAt: 'created_at', // Map createdAt to created_at column
    updatedAt: 'updated_at', // Map updatedAt to updated_at column
  });

  // Define model associations
  Customer.associate = (models) => {
    // Customer has many Sales (one-to-many relationship)
    Customer.hasMany(models.Sale, {
      foreignKey: 'customer_id', // Foreign key in Sales table
      as: 'sales', // Alias for association
    });
    
    // Customer belongs to PriceList (many-to-one relationship, optional)
    Customer.belongsTo(models.PriceList, {
      foreignKey: 'price_list_id', // Foreign key in Customer table
      as: 'priceList', // Alias for association
    });
    
    // Customer has many DiscountRules (one-to-many relationship)
    Customer.hasMany(models.DiscountRule, {
      foreignKey: 'customer_id', // Foreign key in DiscountRules table
      as: 'discountRules', // Alias for association
    });
  };

  // Return Customer model
  return Customer;
};

