/**
 * Application Configuration
 * Central configuration for the Smart POS application
 */

// Load environment variables
require('dotenv').config();

// Application configuration object
const appConfig = {
  // Server configuration
  server: {
    port: process.env.PORT || 3000, // Server port number
    env: process.env.NODE_ENV || 'development', // Current environment
    host: process.env.HOST || 'localhost', // Server host
  },
  
  // JWT configuration for authentication
  jwt: {
    secret: process.env.JWT_SECRET || 'your-secret-key-change-in-production', // JWT secret key
    expiresIn: process.env.JWT_EXPIRES_IN || '24h', // Token expiration time
    refreshExpiresIn: process.env.JWT_REFRESH_EXPIRES_IN || '7d', // Refresh token expiration
  },
  
  // KRA eTIMS configuration
  kra: {
    apiUrl: process.env.KRA_API_URL || 'https://api.etims.kra.go.ke', // KRA API base URL
    apiKey: process.env.KRA_API_KEY || '', // KRA API key
    apiSecret: process.env.KRA_API_SECRET || '', // KRA API secret
    oscuUrl: process.env.KRA_OSCU_URL || '', // OSCU (Offline System Control Unit) URL
    enableFiscalisation: process.env.ENABLE_FISCALISATION === 'true', // Enable/disable fiscalisation
  },
  
  // Payment gateway configuration (Paystack)
  payment: {
    paystackSecretKey: process.env.PAYSTACK_SECRET_KEY || '', // Paystack secret key
    paystackPublicKey: process.env.PAYSTACK_PUBLIC_KEY || '', // Paystack public key
    paystackBaseUrl: process.env.PAYSTACK_BASE_URL || 'https://api.paystack.co', // Paystack API URL
  },
  
  // M-Pesa Daraja API configuration
  mpesa: {
    baseUrl: process.env.MPESA_BASE_URL || 'https://sandbox.safaricom.co.ke', // M-Pesa API base URL
    consumerKey: process.env.MPESA_CONSUMER_KEY || '', // M-Pesa consumer key
    consumerSecret: process.env.MPESA_CONSUMER_SECRET || '', // M-Pesa consumer secret
    passkey: process.env.MPESA_PASSKEY || '', // M-Pesa passkey
    shortcode: process.env.MPESA_SHORTCODE || '', // M-Pesa business shortcode
    callbackUrl: process.env.MPESA_CALLBACK_URL || '', // M-Pesa callback URL
    timeoutUrl: process.env.MPESA_TIMEOUT_URL || '', // M-Pesa timeout URL
    timeoutSeconds: parseInt(process.env.MPESA_TIMEOUT_SECONDS) || 60, // STK Push timeout in seconds
  },
  
  // Barcode/QR code configuration
  barcode: {
    prefix: process.env.BARCODE_PREFIX || 'XYZ', // Prefix for generated barcodes
    length: parseInt(process.env.BARCODE_LENGTH) || 12, // Barcode length
  },
  
  // File upload configuration
  upload: {
    maxFileSize: parseInt(process.env.MAX_FILE_SIZE) || 5242880, // Maximum file size (5MB)
    allowedFileTypes: process.env.ALLOWED_FILE_TYPES?.split(',') || ['image/jpeg', 'image/png', 'image/jpg'], // Allowed file types
    uploadPath: process.env.UPLOAD_PATH || './uploads', // File upload directory
  },
  
  // Rate limiting configuration
  rateLimit: {
    windowMs: parseInt(process.env.RATE_LIMIT_WINDOW_MS) || 15 * 60 * 1000, // Time window in milliseconds
    max: parseInt(process.env.RATE_LIMIT_MAX) || 100, // Maximum requests per window
  },
  
  // CORS configuration
  cors: {
    origin: process.env.CORS_ORIGIN || '*', // Allowed origins
    credentials: process.env.CORS_CREDENTIALS === 'true', // Allow credentials
  },
  
  // Logging configuration
  logging: {
    level: process.env.LOG_LEVEL || 'info', // Log level (error, warn, info, debug)
    format: process.env.LOG_FORMAT || 'combined', // Log format
  },
};

// Export the configuration
module.exports = appConfig;

