# API Testing Guide with cURL - Clothing Industry POS

## Overview
This guide provides comprehensive cURL commands for testing all API endpoints in the XYZ POS Server application, specifically designed for a **clothing manufacturing and retail business**. It covers all phases from Phase 0 to Phase 16, including authentication, CRUD operations, and business logic testing.

**Business Context**: This POS system manages a clothing business that:
- Procures raw materials (fabric, thread, buttons, zipperns)
- Produces finished goods (dresses, shirts, pants) using BOMs
- Sells to both retail customers (B2C) and wholesale customers (B2B)
- Tracks inventory using both quantity-based and UID-based methods
- Integrates with KRA eTIMS for fiscal receipt generation

## Prerequisites

1. **Server Running**: Start the server
   ```bash
   cd server
   npm start
   # or
   npm run dev
   ```

2. **Base URL**: `http://localhost:3000`

3. **Authentication**: Most endpoints require JWT token authentication. Login first to get a token.

4. **Save Token**: Extract and save the JWT token from login response for all subsequent requests
   ```bash
   TOKEN=$(curl -X POST http://localhost:3000/api/auth/login \
     -H "Content-Type: application/json" \
     -d '{"username":"admin","password":"admin123"}' \
     | jq -r '.data.token')
   ```

---

## Phase 0: User Management APIs

### 0.1 Create User (Cashier/Manager/Admin)
```bash
# POST /api/users
# Required: username, email, password (if creating with password)
# Optional: full_name, role, pin_code
# Note: Manager/Admin only - Used to create cashiers, managers, or other admins
curl -X POST http://localhost:3000/api/users \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "username": "cashier1",
    "email": "cashier1@clothingshop.com",
    "password": "cashier123",
    "full_name": "Cashier One",
    "role": "cashier",
    "pin_code": "1234"
  }'
```

### 0.2 Create Manager
```bash
# POST /api/users
curl -X POST http://localhost:3000/api/users \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "username": "manager1",
    "email": "manager1@clothingshop.com",
    "password": "manager123",
    "full_name": "Manager One",
    "role": "manager",
    "pin_code": "5678"
  }'
```

### 0.3 List Users
```bash
# GET /api/users?page=1&limit=50&role=cashier&active=true
curl -X GET "http://localhost:3000/api/users?page=1&limit=50&role=cashier&active=true" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 0.4 Get User by ID
```bash
# GET /api/users/:id
curl -X GET http://localhost:3000/api/users/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 0.5 Update User
```bash
# PUT /api/users/:id
# Users can update themselves, Managers/Admins can update anyone
curl -X PUT http://localhost:3000/api/users/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "full_name": "Updated Cashier One",
    "pin_code": "9999",
    "active": true
  }'
```

### 0.6 Delete User (Soft Delete)
```bash
# DELETE /api/users/:id
# Manager/Admin only
curl -X DELETE http://localhost:3000/api/users/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

---

## Phase 1: Authentication APIs

### 1.1 System Setup (Initial Admin User - Only when no users exist)
```bash
# POST /api/auth/setup
# Required: username, email, password, full_name
# Optional: pin_code
# Note: Only works when database is empty (no existing users)
curl -X POST http://localhost:3000/api/auth/setup \
  -H "Content-Type: application/json" \
  -d '{
    "username": "admin",
    "email": "admin@clothingshop.com",
    "password": "admin123",
    "full_name": "System Administrator",
    "pin_code": "1234"
  }'
```

### 1.2 User Login (Username/Password)
```bash
# POST /api/auth/login
# Required: username, password
curl -X POST http://localhost:3000/api/auth/login \
  -H "Content-Type: application/json" \
  -d '{
    "username": "admin",
    "password": "admin123"
  }'

# Save the token from response for subsequent requests
# Example response: {"success":true,"data":{"token":"eyJhbGc...","user":{...}}}
```

### 1.3 Quick POS Login (Username/PIN)
```bash
# POST /api/auth/login-pin
# Required: username, pin_code
# Note: Faster login for POS cashiers using PIN instead of password
curl -X POST http://localhost:3000/api/auth/login-pin \
  -H "Content-Type: application/json" \
  -d '{
    "username": "cashier1",
    "pin_code": "1234"
  }'
```

### 1.4 Get Current User Profile
```bash
# GET /api/auth/profile
curl -X GET http://localhost:3000/api/auth/profile \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 1.5 Refresh Token (Not Yet Implemented)
```bash
# POST /api/auth/refresh
# Note: This endpoint exists but returns 501 Not Implemented
curl -X POST http://localhost:3000/api/auth/refresh \
  -H "Authorization: Bearer YOUR_REFRESH_TOKEN_HERE"
```

---

## Phase 2: Product APIs (Clothing Industry Examples)

### 2.1 Create Raw Material Product (Fabric)
```bash
# POST /api/products
# Required fields: name, product_type
curl -X POST http://localhost:3000/api/products \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Cotton Fabric",
    "sku": "RM-FAB-COTTON-001",
    "product_type": "RM",
    "track_inventory": true,
    "description": "100% cotton fabric for dress production",
    "default_vat_rate": 16.0
  }'
```

### 2.2 Create Finished Good Product (Dress)
```bash
# POST /api/products
curl -X POST http://localhost:3000/api/products \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Summer Dress",
    "sku": "FG-DRESS-SUMMER-001",
    "product_type": "FG",
    "track_inventory": true,
    "description": "Beautiful summer dress with floral pattern",
    "default_vat_rate": 16.0
  }'
```

### 2.3 Create Product Variant (Dress Size and Color)
```bash
# POST /api/products/:id/variants
# Required: attributes (JSON object)
# Note: Replace PRODUCT_ID with actual dress product ID
curl -X POST http://localhost:3000/api/products/PRODUCT_ID/variants \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "sku": "FG-DRESS-SUMMER-001-M-BLUE",
    "attributes": {
      "size": "M",
      "color": "Blue"
    }
  }'
```

### 2.4 Create Another Variant (Different Size/Color)
```bash
# POST /api/products/:id/variants
curl -X POST http://localhost:3000/api/products/PRODUCT_ID/variants \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "sku": "FG-DRESS-SUMMER-001-L-RED",
    "attributes": {
      "size": "L",
      "color": "Red"
    }
  }'
```

### 2.5 List Products
```bash
# GET /api/products?page=1&limit=50
curl -X GET "http://localhost:3000/api/products?page=1&limit=50" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 2.6 Get Product by ID
```bash
# GET /api/products/:id
curl -X GET http://localhost:3000/api/products/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 2.7 Update Product
```bash
# PUT /api/products/:id
curl -X PUT http://localhost:3000/api/products/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Updated Summer Dress",
    "description": "Updated description for summer dress"
  }'
```

### 2.8 Search Products
```bash
# GET /api/products/search?q=dress&limit=20
curl -X GET "http://localhost:3000/api/products/search?q=dress&limit=20" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 2.9 List Product Variants
```bash
# GET /api/products/:id/variants
curl -X GET http://localhost:3000/api/products/1/variants \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 2.10 Update Product Variant
```bash
# PUT /api/products/:productId/variants/:variantId
curl -X PUT http://localhost:3000/api/products/1/variants/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "sku": "FG-DRESS-SUMMER-001-M-BLUE-UPDATED",
    "attributes": {
      "size": "M",
      "color": "Navy Blue"
    }
  }'
```

### 2.11 Delete Product Variant
```bash
# DELETE /api/products/:productId/variants/:variantId
curl -X DELETE http://localhost:3000/api/products/1/variants/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 2.12 Upload Product Image
```bash
# POST /api/products/:id/image
# Note: Requires multipart/form-data, not JSON
curl -X POST http://localhost:3000/api/products/1/image \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -F "image=@/path/to/product-image.jpg"
```

---

## Phase 3: Inventory APIs (Quantity-Based)

### 3.1 Get Inventory Quantity
```bash
# GET /api/inventory/quantity?product_id=1&variant_id=null
curl -X GET "http://localhost:3000/api/inventory/quantity?product_id=1&variant_id=null" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 3.2 Check Stock Availability
```bash
# GET /api/inventory/check-availability?product_id=1&quantity=10
curl -X GET "http://localhost:3000/api/inventory/check-availability?product_id=1&quantity=10" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 3.3 Adjust Inventory Quantity (Add Raw Material Stock)
```bash
# POST /api/inventory/adjust
# Required: product_id, quantity_change, reason, user_id
curl -X POST http://localhost:3000/api/inventory/adjust \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "product_id": 1,
    "quantity_change": 500,
    "reason": "INITIAL_STOCK",
    "user_id": 1
  }'
```

### 3.4 Set Inventory Quantity
```bash
# PUT /api/inventory/set-quantity
# Required: product_id, quantity
curl -X PUT http://localhost:3000/api/inventory/set-quantity \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "product_id": 1,
    "quantity": 1000
  }'
```

### 3.4 Set Reorder Level
```bash
# PUT /api/inventory/reorder-level
# Required: product_id, reorder_level
curl -X PUT http://localhost:3000/api/inventory/reorder-level \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "product_id": 1,
    "reorder_level": 100
  }'
```

### 3.5 List Inventory
```bash
# GET /api/inventory?page=1&limit=50
curl -X GET "http://localhost:3000/api/inventory?page=1&limit=50" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 3.6 Get Low Stock Items
```bash
# GET /api/inventory/low-stock
curl -X GET http://localhost:3000/api/inventory/low-stock \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 3.7 Get Inventory Movements
```bash
# GET /api/inventory/movements?product_id=1
curl -X GET "http://localhost:3000/api/inventory/movements?product_id=1" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

---

## Phase 6: Inventory APIs (UID-Based)

### 6.1 Create Inventory Item (UID) for Finished Good
```bash
# POST /api/inventory/items
# Required: product_id, source
# Optional: variant_id, uid, barcode, status
# Note: UID and barcode will be auto-generated if not provided
curl -X POST http://localhost:3000/api/inventory/items \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "product_id": 2,
    "variant_id": 1,
    "source": "PRODUCTION",
    "status": "IN_STOCK"
  }'
```

### 6.2 Get Inventory Item by UID/Barcode
```bash
# GET /api/inventory/items/:uid
# Note: :uid can be either UID or barcode
curl -X GET http://localhost:3000/api/inventory/items/UID-1234567890 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 6.3 Scan Inventory Item (Returns QR Code Data)
```bash
# GET /api/inventory/items/:uid/scan
# Note: Returns inventory item with QR code data for scanning
curl -X GET http://localhost:3000/api/inventory/items/UID-1234567890/scan \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 6.4 Update Inventory Item Status
```bash
# PUT /api/inventory/items/:uid/status
# Required: status
# Optional: reason
curl -X PUT http://localhost:3000/api/inventory/items/UID-1234567890/status \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "status": "SOLD",
    "reason": "Sold to customer"
  }'
```

### 6.5 List Inventory Items
```bash
# GET /api/inventory/items?status=IN_STOCK&product_id=2&page=1&limit=50
curl -X GET "http://localhost:3000/api/inventory/items?status=IN_STOCK&product_id=2&page=1&limit=50" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

---

## Phase 4: Sales & POS APIs

### 4.1 Create POS Sale (Walk-in Customer)
```bash
# POST /api/sales
# Required: items (array with product_id, quantity)
# Optional: sale_type (default: POS), customer_id, discount_code, items.*.unit_price, items.*.variant_id
curl -X POST http://localhost:3000/api/sales \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "sale_type": "POS",
    "items": [
      {
        "product_id": 2,
        "variant_id": 1,
        "quantity": 2,
        "unit_price": 2500.00
      }
    ],
    "customer_id": null
  }'
```

### 4.2 Create Invoice Sale (B2B Customer)
```bash
# POST /api/sales
curl -X POST http://localhost:3000/api/sales \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "sale_type": "INVOICE",
    "customer_id": 1,
    "items": [
      {
        "product_id": 2,
        "variant_id": 1,
        "quantity": 10,
        "unit_price": 2000.00
      },
      {
        "product_id": 2,
        "variant_id": 2,
        "quantity": 5,
        "unit_price": 2200.00
      }
    ],
    "discount_code": "SUMMER2024"
  }'
```

### 4.3 Get Sale by ID
```bash
# GET /api/sales/:id
curl -X GET http://localhost:3000/api/sales/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 4.4 List Sales
```bash
# GET /api/sales?page=1&limit=50&status=COMPLETED
curl -X GET "http://localhost:3000/api/sales?page=1&limit=50&status=COMPLETED" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 4.5 Cancel Sale
```bash
# POST /api/sales/:id/cancel
curl -X POST http://localhost:3000/api/sales/1/cancel \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "reason": "Customer cancelled"
  }'
```

---

## Phase 5: Payment APIs

### 5.1 Create Payment (Cash/Bank)
```bash
# POST /api/payments
# Required: sale_id, provider, amount
# Optional: reference
# Note: provider can be CASH, BANK, PAYSTACK, or MOBILE
curl -X POST http://localhost:3000/api/payments \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "sale_id": 1,
    "provider": "CASH",
    "amount": 5000.00,
    "reference": "CASH-001"
  }'
```

### 5.2 Initialize Paystack Payment
```bash
# POST /api/payments/paystack/initialize
# Required: sale_id, email, amount
# Optional: metadata
curl -X POST http://localhost:3000/api/payments/paystack/initialize \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "sale_id": 1,
    "email": "customer@example.com",
    "amount": 5000.00
  }'
```

### 5.3 Verify Paystack Payment
```bash
# POST /api/payments/paystack/verify/:reference
# Note: Reference is in URL path, not body
curl -X POST http://localhost:3000/api/payments/paystack/verify/PAYSTACK-REF-123 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 5.4 Paystack Webhook
```bash
# POST /api/payments/paystack/webhook
# Note: This is called by Paystack, not typically used in manual testing
curl -X POST http://localhost:3000/api/payments/paystack/webhook \
  -H "Content-Type: application/json" \
  -d '{
    "event": "charge.success",
    "data": {
      "reference": "PAYSTACK-REF-123"
    }
  }'
```

### 5.5 Process Mobile Money Payment
```bash
# POST /api/payments/mobile
# Required: sale_id, amount, phone_number
curl -X POST http://localhost:3000/api/payments/mobile \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "sale_id": 1,
    "amount": 5000.00,
    "phone_number": "+254712345678"
  }'
```

### 5.6 Get Payment by ID
```bash
# GET /api/payments/:id
curl -X GET http://localhost:3000/api/payments/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 5.7 List Payments
```bash
# GET /api/payments?sale_id=1
curl -X GET "http://localhost:3000/api/payments?sale_id=1" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 5.8 Get Sale Payments
```bash
# GET /api/payments/sale/:saleId
curl -X GET http://localhost:3000/api/payments/sale/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

---

## Phase 7A: Category APIs

### 7A.1 Create Category
```bash
# POST /api/categories
# Required: name
# Optional: description, parent_id
curl -X POST http://localhost:3000/api/categories \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Women\'s Clothing",
    "description": "Women clothing products",
    "parent_id": null
  }'
```

### 7A.2 Create Subcategory
```bash
# POST /api/categories
curl -X POST http://localhost:3000/api/categories \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Dresses",
    "description": "Dress category",
    "parent_id": 1
  }'
```

### 7A.3 Get Category by ID
```bash
# GET /api/categories/:id
curl -X GET http://localhost:3000/api/categories/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7A.4 List Categories
```bash
# GET /api/categories
curl -X GET http://localhost:3000/api/categories \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7A.5 Get Category Hierarchy (Tree Structure)
```bash
# GET /api/categories/hierarchy?root_category_id=1
curl -X GET "http://localhost:3000/api/categories/hierarchy?root_category_id=1" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7A.6 Get Product Categories
```bash
# GET /api/categories/products/:productId
curl -X GET http://localhost:3000/api/categories/products/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7A.7 Assign Product to Categories
```bash
# POST /api/categories/products/:productId
# Required: category_ids (array)
# Optional: primary_category_id
curl -X POST http://localhost:3000/api/categories/products/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "category_ids": [1, 2],
    "primary_category_id": 1
  }'
```

### 7A.8 Remove Product from Category
```bash
# DELETE /api/categories/products/:productId/:categoryId
curl -X DELETE http://localhost:3000/api/categories/products/1/2 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7A.9 Update Category
```bash
# PUT /api/categories/:id
curl -X PUT http://localhost:3000/api/categories/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Updated Women\'s Clothing",
    "description": "Updated description"
  }'
```

### 7A.10 Delete Category
```bash
# DELETE /api/categories/:id
curl -X DELETE http://localhost:3000/api/categories/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

---

## Phase 7B: Pricing APIs

### 7B.1 Create Price List
```bash
# POST /api/pricing/price-lists
# Required: name, code, price_list_type
# Optional: currency (default: KES), description, is_default
curl -X POST http://localhost:3000/api/pricing/price-lists \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Retail Price List",
    "code": "RETAIL-PL-001",
    "price_list_type": "RETAIL",
    "currency": "KES",
    "description": "Default retail pricing",
    "is_default": true
  }'
```

### 7B.2 Get Default Price List
```bash
# GET /api/pricing/price-lists/default
curl -X GET http://localhost:3000/api/pricing/price-lists/default \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7B.3 List Price Lists
```bash
# GET /api/pricing/price-lists?price_list_type=RETAIL&active=true
curl -X GET "http://localhost:3000/api/pricing/price-lists?price_list_type=RETAIL&active=true" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7B.4 Get Price List by ID
```bash
# GET /api/pricing/price-lists/:id
curl -X GET http://localhost:3000/api/pricing/price-lists/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7B.5 Update Price List
```bash
# PUT /api/pricing/price-lists/:id
curl -X PUT http://localhost:3000/api/pricing/price-lists/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Updated Retail Price List",
    "is_default": false,
    "active": true
  }'
```

### 7B.6 Delete Price List
```bash
# DELETE /api/pricing/price-lists/:id
curl -X DELETE http://localhost:3000/api/pricing/price-lists/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7B.7 Create Product Price
```bash
# POST /api/pricing/prices
# Required: price_list_id, product_id, price
# Optional: variant_id, min_quantity, max_quantity, effective_from, effective_to, active
curl -X POST http://localhost:3000/api/pricing/prices \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "price_list_id": 1,
    "product_id": 2,
    "variant_id": null,
    "price": 2500.00,
    "min_quantity": 1,
    "active": true
  }'
```

### 7B.8 List Product Prices
```bash
# GET /api/pricing/prices?price_list_id=1&product_id=2&active=true
curl -X GET "http://localhost:3000/api/pricing/prices?price_list_id=1&product_id=2&active=true" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7B.9 Get Price (Lookup)
```bash
# GET /api/pricing/prices/lookup?product_id=1&quantity=10&price_list_id=1
curl -X GET "http://localhost:3000/api/pricing/prices/lookup?product_id=1&quantity=10&price_list_id=1" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7B.10 Get Price for Customer
```bash
# GET /api/pricing/prices/customer?product_id=1&quantity=10&customer_id=1
curl -X GET "http://localhost:3000/api/pricing/prices/customer?product_id=1&quantity=10&customer_id=1" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7B.11 Get Product Price by ID
```bash
# GET /api/pricing/prices/:id
curl -X GET http://localhost:3000/api/pricing/prices/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7B.12 Update Product Price
```bash
# PUT /api/pricing/prices/:id
curl -X PUT http://localhost:3000/api/pricing/prices/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "price": 2700.00,
    "active": true
  }'
```

### 7B.13 Delete Product Price
```bash
# DELETE /api/pricing/prices/:id
curl -X DELETE http://localhost:3000/api/pricing/prices/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

---

## Phase 7C: Tax APIs

### 7C.1 Create Tax Rate
```bash
# POST /api/taxes/tax-rates
# Required: code, name, rate
# Optional: is_default
curl -X POST http://localhost:3000/api/taxes/tax-rates \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "code": "VAT",
    "name": "Value Added Tax",
    "rate": 16.0,
    "is_default": true
  }'
```

### 7C.2 List Tax Rates
```bash
# GET /api/taxes/tax-rates?tax_type=VAT&active=true
curl -X GET "http://localhost:3000/api/taxes/tax-rates?tax_type=VAT&active=true" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7C.3 Get Tax Rate by ID
```bash
# GET /api/taxes/tax-rates/:id
curl -X GET http://localhost:3000/api/taxes/tax-rates/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7C.4 Get Tax Rate by Code
```bash
# GET /api/taxes/tax-rates/code/:code
curl -X GET http://localhost:3000/api/taxes/tax-rates/code/VAT \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7C.5 Update Tax Rate
```bash
# PUT /api/taxes/tax-rates/:id
curl -X PUT http://localhost:3000/api/taxes/tax-rates/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "rate": 16.0,
    "active": true
  }'
```

### 7C.6 Delete Tax Rate
```bash
# DELETE /api/taxes/tax-rates/:id
curl -X DELETE http://localhost:3000/api/taxes/tax-rates/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7C.7 Create Tax Category
```bash
# POST /api/taxes/tax-categories
# Required: name, tax_rate_id
curl -X POST http://localhost:3000/api/taxes/tax-categories \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Standard VAT",
    "tax_rate_id": 1
  }'
```

### 7C.8 List Tax Categories
```bash
# GET /api/taxes/tax-categories?tax_rate_id=1
curl -X GET "http://localhost:3000/api/taxes/tax-categories?tax_rate_id=1" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7C.9 Get Tax Category by ID
```bash
# GET /api/taxes/tax-categories/:id
curl -X GET http://localhost:3000/api/taxes/tax-categories/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7C.10 Update Tax Category
```bash
# PUT /api/taxes/tax-categories/:id
curl -X PUT http://localhost:3000/api/taxes/tax-categories/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Updated Standard VAT"
  }'
```

### 7C.11 Delete Tax Category
```bash
# DELETE /api/taxes/tax-categories/:id
curl -X DELETE http://localhost:3000/api/taxes/tax-categories/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7C.12 Create Tax Exemption
```bash
# POST /api/taxes/tax-exemptions
# Required: tax_category_id, entity_type (CUSTOMER or PRODUCT)
# Required (conditional): customer_id if entity_type=CUSTOMER, product_id if entity_type=PRODUCT
# Optional: certificate_number, effective_from, effective_to, active
curl -X POST http://localhost:3000/api/taxes/tax-exemptions \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "tax_category_id": 1,
    "entity_type": "CUSTOMER",
    "customer_id": 1,
    "certificate_number": "EXEMPT-001",
    "active": true
  }'
```

### 7C.13 List Tax Exemptions
```bash
# GET /api/taxes/tax-exemptions?entity_type=CUSTOMER&customer_id=1
curl -X GET "http://localhost:3000/api/taxes/tax-exemptions?entity_type=CUSTOMER&customer_id=1" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7C.14 Get Tax Exemption by ID
```bash
# GET /api/taxes/tax-exemptions/:id
curl -X GET http://localhost:3000/api/taxes/tax-exemptions/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7C.15 Update Tax Exemption
```bash
# PUT /api/taxes/tax-exemptions/:id
curl -X PUT http://localhost:3000/api/taxes/tax-exemptions/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "active": false
  }'
```

### 7C.16 Delete Tax Exemption
```bash
# DELETE /api/taxes/tax-exemptions/:id
curl -X DELETE http://localhost:3000/api/taxes/tax-exemptions/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7C.17 Calculate Product Tax
```bash
# GET /api/taxes/calculate?product_id=1&quantity=10&unit_price=1000&customer_id=1
curl -X GET "http://localhost:3000/api/taxes/calculate?product_id=1&quantity=10&unit_price=1000&customer_id=1" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 7C.18 Get Product Tax Rate
```bash
# GET /api/taxes/tax-rate?product_id=1&customer_id=1
curl -X GET "http://localhost:3000/api/taxes/tax-rate?product_id=1&customer_id=1" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

---

## Phase 8: Discount APIs

### 8.1 Create Discount
```bash
# POST /api/discounts
# Required: name, discount_type, discount_value
# Optional: code, min_purchase_amount, max_discount_amount, effective_from, effective_to, max_uses_per_customer, max_total_uses, active, description, rules
curl -X POST http://localhost:3000/api/discounts \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Summer Sale",
    "code": "SUMMER2024",
    "discount_type": "PERCENTAGE",
    "discount_value": 10.0,
    "max_discount_amount": 1000.00,
    "effective_from": "2024-01-01T00:00:00Z",
    "effective_to": "2024-12-31T23:59:59Z",
    "is_active": true
  }'
```

### 8.2 List Discounts
```bash
# GET /api/discounts?is_active=true&discount_type=PERCENTAGE&search=summer
curl -X GET "http://localhost:3000/api/discounts?is_active=true&discount_type=PERCENTAGE&search=summer" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 8.3 Validate Discount
```bash
# POST /api/discounts/validate
# Required: discount_code or discount_id
# Optional: customer_id
curl -X POST http://localhost:3000/api/discounts/validate \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "discount_code": "SUMMER2024",
    "customer_id": 1
  }'
```

### 8.4 Apply Discount (Preview Calculation)
```bash
# POST /api/discounts/apply
# Required: discount_code or discount_id, sale_items (array)
# Optional: customer_id
curl -X POST http://localhost:3000/api/discounts/apply \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "discount_code": "SUMMER2024",
    "customer_id": 1,
    "sale_items": [
      {
        "product_id": 1,
        "quantity": 2,
        "unit_price": 1000.00
      }
    ]
  }'
```

### 8.5 Get Discount by ID
```bash
# GET /api/discounts/:id
curl -X GET http://localhost:3000/api/discounts/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 8.6 Get Discount by Code
```bash
# GET /api/discounts/code/:code
curl -X GET http://localhost:3000/api/discounts/code/SUMMER2024 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 8.7 Update Discount
```bash
# PUT /api/discounts/:id
curl -X PUT http://localhost:3000/api/discounts/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "is_active": false
  }'
```

### 8.8 Delete Discount
```bash
# DELETE /api/discounts/:id
curl -X DELETE http://localhost:3000/api/discounts/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 8.9 Create Discount Rule
```bash
# POST /api/discounts/:discountId/rules
# Required: rule_type (PRODUCT, CATEGORY, CUSTOMER, ALL)
# Required (conditional): product_id if PRODUCT, category_id if CATEGORY, customer_id if CUSTOMER
# Optional: min_quantity
curl -X POST http://localhost:3000/api/discounts/1/rules \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "rule_type": "PRODUCT",
    "product_id": 1,
    "min_quantity": 2
  }'
```

### 8.10 List Discount Rules
```bash
# GET /api/discounts/:discountId/rules
curl -X GET http://localhost:3000/api/discounts/1/rules \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 8.11 Get Discount Rule by ID
```bash
# GET /api/discounts/:discountId/rules/:ruleId
curl -X GET http://localhost:3000/api/discounts/1/rules/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 8.12 Update Discount Rule
```bash
# PUT /api/discounts/:discountId/rules/:ruleId
curl -X PUT http://localhost:3000/api/discounts/1/rules/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "min_quantity": 5
  }'
```

### 8.13 Delete Discount Rule
```bash
# DELETE /api/discounts/:discountId/rules/:ruleId
curl -X DELETE http://localhost:3000/api/discounts/1/rules/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

---

## Phase 9: Procurement APIs

### 9.1 Create Vendor (Fabric Supplier)
```bash
# POST /api/procurement/vendors
# Required: name
# Optional: type (default: REGISTERED)
curl -X POST http://localhost:3000/api/procurement/vendors \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "ABC Textile Suppliers",
    "type": "REGISTERED"
  }'
```

### 9.2 List Vendors
```bash
# GET /api/procurement/vendors?type=REGISTERED&search=ABC
curl -X GET "http://localhost:3000/api/procurement/vendors?type=REGISTERED&search=ABC" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 9.3 Get Vendor by ID
```bash
# GET /api/procurement/vendors/:id
curl -X GET http://localhost:3000/api/procurement/vendors/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 9.4 Update Vendor
```bash
# PUT /api/procurement/vendors/:id
curl -X PUT http://localhost:3000/api/procurement/vendors/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Updated ABC Textile Suppliers"
  }'
```

### 9.5 Delete Vendor
```bash
# DELETE /api/procurement/vendors/:id
curl -X DELETE http://localhost:3000/api/procurement/vendors/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 9.6 Create Purchase Order
```bash
# POST /api/procurement/purchase-orders
# Required: vendor_id
# Optional: status (default: DRAFT), expected_delivery_date
# Note: items array should be included but PO doesn't directly require it - items are added via GRN
curl -X POST http://localhost:3000/api/procurement/purchase-orders \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "vendor_id": 1,
    "status": "DRAFT",
    "expected_delivery_date": "2024-12-31"
  }'
```

### 9.7 List Purchase Orders
```bash
# GET /api/procurement/purchase-orders?vendor_id=1&status=DRAFT
curl -X GET "http://localhost:3000/api/procurement/purchase-orders?vendor_id=1&status=DRAFT" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 9.8 Get Purchase Order by ID
```bash
# GET /api/procurement/purchase-orders/:id
curl -X GET http://localhost:3000/api/procurement/purchase-orders/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 9.9 Update Purchase Order
```bash
# PUT /api/procurement/purchase-orders/:id
curl -X PUT http://localhost:3000/api/procurement/purchase-orders/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "status": "CONFIRMED"
  }'
```

### 9.10 Update Purchase Order Status
```bash
# PUT /api/procurement/purchase-orders/:id/status
curl -X PUT http://localhost:3000/api/procurement/purchase-orders/1/status \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "status": "CLOSED"
  }'
```

### 9.11 Delete Purchase Order
```bash
# DELETE /api/procurement/purchase-orders/:id
curl -X DELETE http://localhost:3000/api/procurement/purchase-orders/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 9.12 Create GRN (Goods Receipt Note)
```bash
# POST /api/procurement/grns
# Required: items (array with product_id, quantity, unit_cost)
# Optional: vendor_id, purchase_order_id, received_at, generateUIDs
curl -X POST http://localhost:3000/api/procurement/grns \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "vendor_id": 1,
    "items": [
      {
        "product_id": 1,
        "variant_id": null,
        "quantity": 500,
        "unit_cost": 150.00
      },
      {
        "product_id": 3,
        "variant_id": null,
        "quantity": 1000,
        "unit_cost": 5.00
      }
    ],
    "received_at": "2024-01-15T10:00:00Z",
    "generateUIDs": false
  }'
```

### 9.13 Process GRN (Add to Inventory)
```bash
# POST /api/procurement/grns/:id/process
# Optional: generateUIDs (body parameter)
curl -X POST http://localhost:3000/api/procurement/grns/1/process \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "generateUIDs": false
  }'
```

### 9.6 Get GRN
```bash
# GET /api/procurement/grns/:id
curl -X GET http://localhost:3000/api/procurement/grns/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 9.7 List GRNs
```bash
# GET /api/procurement/grns?vendor_id=1&page=1&limit=50
curl -X GET "http://localhost:3000/api/procurement/grns?vendor_id=1&page=1&limit=50" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

---

## Phase 10: Production APIs

### 10.1 Create BOM (Bill of Materials)
```bash
# POST /api/production/boms
# Required: fg_variant_id, items (array with rm_product_id, quantity_per_unit)
# Example: Summer Dress (Size M, Blue) requires 2m cotton fabric, 1 zipper, 0.5m thread
curl -X POST http://localhost:3000/api/production/boms \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "fg_variant_id": 1,
    "items": [
      {
        "rm_product_id": 1,
        "quantity_per_unit": 2.0
      },
      {
        "rm_product_id": 3,
        "quantity_per_unit": 1.0
      },
      {
        "rm_product_id": 4,
        "quantity_per_unit": 0.5
      }
    ]
  }'
```

### 10.2 Get BOM by ID
```bash
# GET /api/production/boms/:id
curl -X GET http://localhost:3000/api/production/boms/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 10.3 Get BOM by Variant ID
```bash
# GET /api/production/boms/variant/:variantId
curl -X GET http://localhost:3000/api/production/boms/variant/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 10.4 List BOMs
```bash
# GET /api/production/boms?fg_variant_id=1&page=1&limit=50
curl -X GET "http://localhost:3000/api/production/boms?fg_variant_id=1&page=1&limit=50" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 10.5 Update BOM
```bash
# PUT /api/production/boms/:id
# Optional: items (array) - replaces all BOM items
curl -X PUT http://localhost:3000/api/production/boms/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "items": [
      {
        "rm_product_id": 1,
        "quantity_per_unit": 2.5
      }
    ]
  }'
```

### 10.6 Delete BOM
```bash
# DELETE /api/production/boms/:id
curl -X DELETE http://localhost:3000/api/production/boms/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 10.7 Create Production Order
```bash
# POST /api/production/production-orders
# Required: fg_variant_id, quantity
# Optional: status (default: DRAFT)
# Note: BOM must exist for the fg_variant_id before creating production order
curl -X POST http://localhost:3000/api/production/production-orders \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "fg_variant_id": 1,
    "quantity": 100,
    "status": "DRAFT"
  }'
```

### 10.8 Check Raw Material Availability
```bash
# GET /api/production/production-orders/:id/check-availability
curl -X GET http://localhost:3000/api/production/production-orders/1/check-availability \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 10.9 Confirm Production (Consume Raw Materials, Create Finished Goods)
```bash
# POST /api/production/production-orders/:id/confirm
# Optional: generateUIDs (default: false)
# This will: consume raw materials from inventory and create finished goods
curl -X POST http://localhost:3000/api/production/production-orders/1/confirm \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "generateUIDs": true
  }'
```

### 10.10 Get Production Order
```bash
# GET /api/production/production-orders/:id
curl -X GET http://localhost:3000/api/production/production-orders/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 10.11 List Production Orders
```bash
# GET /api/production/production-orders?fg_variant_id=1&status=DRAFT&page=1&limit=50
curl -X GET "http://localhost:3000/api/production/production-orders?fg_variant_id=1&status=DRAFT&page=1&limit=50" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

---

## Phase 11: Returns APIs

### 11.1 Create Return
```bash
# POST /api/returns
# Required: sale_id, items (array with sale_item_id, inventory_item_id, quantity)
# Optional: reason, items.*.reason
# Note: inventory_item_id is required for UID-tracked items
curl -X POST http://localhost:3000/api/returns \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "sale_id": 1,
    "reason": "Customer returned defective item",
    "items": [
      {
        "sale_item_id": 1,
        "inventory_item_id": 1,
        "quantity": 1,
        "reason": "Defective dress"
      }
    ]
  }'
```

### 11.2 Complete Return (Update Inventory, Generate Credit Note)
```bash
# POST /api/returns/:id/complete
# This will: update inventory status, create inventory movements, generate fiscal receipt (credit note)
curl -X POST http://localhost:3000/api/returns/1/complete \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 11.3 Get Return
```bash
# GET /api/returns/:id
curl -X GET http://localhost:3000/api/returns/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 11.4 List Returns
```bash
# GET /api/returns?sale_id=1&status=COMPLETED&page=1&limit=50
curl -X GET "http://localhost:3000/api/returns?sale_id=1&status=COMPLETED&page=1&limit=50" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

---

## Phase 12: Shift APIs

### 12.1 Open Shift
```bash
# POST /api/shifts
# Required: user_id
# Optional: opening_cash (default: 0), notes
# Note: user_id must match authenticated user
curl -X POST http://localhost:3000/api/shifts \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "user_id": 1,
    "opening_cash": 10000.00,
    "notes": "Morning shift"
  }'
```

### 12.2 Get Current Shift
```bash
# GET /api/shifts/current
curl -X GET http://localhost:3000/api/shifts/current \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 12.3 List Shifts
```bash
# GET /api/shifts?user_id=1&status=OPEN&page=1&limit=50
curl -X GET "http://localhost:3000/api/shifts?user_id=1&status=OPEN&page=1&limit=50" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 12.4 Get Shift by ID
```bash
# GET /api/shifts/:id
curl -X GET http://localhost:3000/api/shifts/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 12.5 Close Shift
```bash
# POST /api/shifts/:id/close
# Required: actual_closing_cash
# Optional: notes
curl -X POST http://localhost:3000/api/shifts/1/close \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "actual_closing_cash": 50000.00,
    "notes": "End of shift"
  }'
```

### 12.6 Create Cash Count (Mid-Shift)
```bash
# POST /api/shifts/:shiftId/cash-counts
# Required: amount
# Optional: notes
curl -X POST http://localhost:3000/api/shifts/1/cash-counts \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "amount": 45000.00,
    "notes": "Mid-day cash count"
  }'
```

### 12.7 List Cash Counts
```bash
# GET /api/shifts/cash-counts?shift_id=1&count_type=MID_SHIFT
curl -X GET "http://localhost:3000/api/shifts/cash-counts?shift_id=1&count_type=MID_SHIFT" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 12.8 Get Cash Count by ID
```bash
# GET /api/shifts/cash-counts/:id
curl -X GET http://localhost:3000/api/shifts/cash-counts/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

---

## Phase 13: KRA eTIMS APIs

### 13.1 Create Fiscal Receipt (Manual)
```bash
# POST /api/kra-etims/receipts
# Required: receipt_category (SALE or CREDIT_NOTE), receipt_type (B2C or B2B)
# Required (conditional): sale_id if receipt_category=SALE, return_id if receipt_category=CREDIT_NOTE
# Optional: customer_pin (for B2B)
curl -X POST http://localhost:3000/api/kra-etims/receipts \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "receipt_category": "SALE",
    "receipt_type": "B2C",
    "sale_id": 1
  }'
```

### 13.2 List Fiscal Receipts
```bash
# GET /api/kra-etims/receipts?sale_id=1&receipt_category=SALE&fiscal_status=FISCALISED
curl -X GET "http://localhost:3000/api/kra-etims/receipts?sale_id=1&receipt_category=SALE&fiscal_status=FISCALISED" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 13.3 Get Fiscal Receipt by ID
```bash
# GET /api/kra-etims/receipts/:id
curl -X GET http://localhost:3000/api/kra-etims/receipts/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 13.4 Fiscalise Sale
```bash
# POST /api/kra-etims/sales/:saleId/fiscalise
# Optional: customer_pin (for B2B), async (default: false)
# This will: create fiscal receipt, call KRA API, store QR code and receipt number
curl -X POST http://localhost:3000/api/kra-etims/sales/1/fiscalise \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "async": false
  }'
```

### 13.5 Fiscalise Return (Credit Note)
```bash
# POST /api/kra-etims/returns/:returnId/fiscalise
# Optional: customer_pin (for B2B), async (default: false)
# This will: create fiscal receipt (credit note) for the return
curl -X POST http://localhost:3000/api/kra-etims/returns/1/fiscalise \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "async": false
  }'
```

---

## Phase 14: Customer APIs

### 14.1 Create B2C Customer (Walk-in)
```bash
# POST /api/customers
# Required: name
# Optional: customer_type (default: B2C), kra_pin, credit_limit, price_list_id
curl -X POST http://localhost:3000/api/customers \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Jane Doe",
    "customer_type": "B2C",
    "kra_pin": null,
    "credit_limit": 0,
    "price_list_id": null
  }'
```

### 14.2 Create B2B Customer (Wholesale)
```bash
# POST /api/customers
curl -X POST http://localhost:3000/api/customers \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Fashion Retailers Ltd",
    "customer_type": "B2B",
    "kra_pin": "P123456789A",
    "credit_limit": 500000.00,
    "price_list_id": 1
  }'
```

### 14.3 Get Customer
```bash
# GET /api/customers/:id
curl -X GET http://localhost:3000/api/customers/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 14.4 Update Customer
```bash
# PUT /api/customers/:id
curl -X PUT http://localhost:3000/api/customers/1 \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Updated Fashion Retailers Ltd",
    "credit_limit": 600000.00,
    "active": true
  }'
```

### 14.5 Search Customers
```bash
# GET /api/customers/search?q=Fashion&limit=20
curl -X GET "http://localhost:3000/api/customers/search?q=Fashion&limit=20" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 14.6 List Customers
```bash
# GET /api/customers?customer_type=B2B&active=true&page=1&limit=50
curl -X GET "http://localhost:3000/api/customers?customer_type=B2B&active=true&page=1&limit=50" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

---

## Phase 15: Report APIs

### 15.1 Sales Report by Date
```bash
# GET /api/reports/sales/date?start_date=2024-01-01&end_date=2024-12-31
curl -X GET "http://localhost:3000/api/reports/sales/date?start_date=2024-01-01&end_date=2024-12-31" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 15.2 Sales Report by Product
```bash
# GET /api/reports/sales/product?start_date=2024-01-01&end_date=2024-12-31
curl -X GET "http://localhost:3000/api/reports/sales/product?start_date=2024-01-01&end_date=2024-12-31" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 15.3 Sales Report by Customer
```bash
# GET /api/reports/sales/customer?start_date=2024-01-01&end_date=2024-12-31
curl -X GET "http://localhost:3000/api/reports/sales/customer?start_date=2024-01-01&end_date=2024-12-31" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 15.4 Sales Report by Cashier
```bash
# GET /api/reports/sales/cashier?start_date=2024-01-01&end_date=2024-12-31
curl -X GET "http://localhost:3000/api/reports/sales/cashier?start_date=2024-01-01&end_date=2024-12-31" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 15.5 Fast-Moving Products Report
```bash
# GET /api/reports/inventory/fast-moving?limit=20&start_date=2024-01-01&end_date=2024-12-31
curl -X GET "http://localhost:3000/api/reports/inventory/fast-moving?limit=20&start_date=2024-01-01&end_date=2024-12-31" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 15.6 Slow-Moving Products Report
```bash
# GET /api/reports/inventory/slow-moving?limit=20&start_date=2024-01-01&end_date=2024-12-31
curl -X GET "http://localhost:3000/api/reports/inventory/slow-moving?limit=20&start_date=2024-01-01&end_date=2024-12-31" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 15.7 Low Stock Items Report
```bash
# GET /api/reports/inventory/low-stock
curl -X GET http://localhost:3000/api/reports/inventory/low-stock \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 15.8 Profit/Loss Report
```bash
# GET /api/reports/financial/profit-loss?start_date=2024-01-01&end_date=2024-12-31
curl -X GET "http://localhost:3000/api/reports/financial/profit-loss?start_date=2024-01-01&end_date=2024-12-31" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 15.9 Tax Report
```bash
# GET /api/reports/tax?start_date=2024-01-01&end_date=2024-12-31&customer_type=B2B
curl -X GET "http://localhost:3000/api/reports/tax?start_date=2024-01-01&end_date=2024-12-31&customer_type=B2B" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

### 15.10 Shift Report
```bash
# GET /api/reports/shifts?start_date=2024-01-01&end_date=2024-12-31&user_id=1
curl -X GET "http://localhost:3000/api/reports/shifts?start_date=2024-01-01&end_date=2024-12-31&user_id=1" \
  -H "Authorization: Bearer YOUR_TOKEN_HERE"
```

---

## Complete Clothing Business Workflow Example

This section demonstrates a complete business workflow from raw material procurement to final sale.

### Step 1: Setup and Authentication
```bash
# 1. Login to get token
TOKEN=$(curl -X POST http://localhost:3000/api/auth/login \
  -H "Content-Type: application/json" \
  -d '{"username":"admin","password":"admin123"}' \
  | jq -r '.data.token')

echo "Token: $TOKEN"
```

### Step 2: Create Raw Material Products
```bash
# 2. Create Cotton Fabric (Raw Material)
COTTON_FABRIC_ID=$(curl -X POST http://localhost:3000/api/products \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Cotton Fabric",
    "sku": "RM-FAB-COTTON-001",
    "product_type": "RM",
    "track_inventory": true,
    "description": "100% cotton fabric",
    "default_vat_rate": 16.0
  }' | jq -r '.data.id')

# 3. Create Zipper (Raw Material)
ZIPPER_ID=$(curl -X POST http://localhost:3000/api/products \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Zipper",
    "sku": "RM-ZIP-001",
    "product_type": "RM",
    "track_inventory": true,
    "description": "Standard zipper",
    "default_vat_rate": 16.0
  }' | jq -r '.data.id')

echo "Raw Materials Created: Cotton Fabric ID=$COTTON_FABRIC_ID, Zipper ID=$ZIPPER_ID"
```

### Step 3: Create Finished Good Product with Variants
```bash
# 4. Create Summer Dress (Finished Good)
DRESS_ID=$(curl -X POST http://localhost:3000/api/products \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Summer Dress",
    "sku": "FG-DRESS-SUMMER-001",
    "product_type": "FG",
    "track_inventory": true,
    "description": "Beautiful summer dress",
    "default_vat_rate": 16.0
  }' | jq -r '.data.id')

# 5. Create Dress Variant (Size M, Blue)
DRESS_VARIANT_ID=$(curl -X POST http://localhost:3000/api/products/$DRESS_ID/variants \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "sku": "FG-DRESS-SUMMER-001-M-BLUE",
    "attributes": {
      "size": "M",
      "color": "Blue"
    }
  }' | jq -r '.data.id')

echo "Finished Good Created: Dress ID=$DRESS_ID, Variant ID=$DRESS_VARIANT_ID"
```

### Step 4: Create Vendor and Receive Raw Materials (GRN)
```bash
# 6. Create Vendor
VENDOR_ID=$(curl -X POST http://localhost:3000/api/procurement/vendors \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "ABC Textile Suppliers",
    "type": "REGISTERED"
  }' | jq -r '.data.id')

# 7. Create GRN (Goods Receipt Note) for Raw Materials
GRN_ID=$(curl -X POST http://localhost:3000/api/procurement/grns \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d "{
    \"vendor_id\": $VENDOR_ID,
    \"items\": [
      {
        \"product_id\": $COTTON_FABRIC_ID,
        \"quantity\": 1000,
        \"unit_cost\": 150.00
      },
      {
        \"product_id\": $ZIPPER_ID,
        \"quantity\": 500,
        \"unit_cost\": 50.00
      }
    ]
  }" | jq -r '.data.id')

# 8. Process GRN (Add to Inventory)
curl -X POST http://localhost:3000/api/procurement/grns/$GRN_ID/process \
  -H "Authorization: Bearer $TOKEN"

echo "GRN Processed: $GRN_ID"
```

### Step 5: Create BOM (Bill of Materials)
```bash
# 9. Create BOM for Summer Dress
# Recipe: 1 dress requires 2m cotton fabric + 1 zipper
BOM_ID=$(curl -X POST http://localhost:3000/api/production/boms \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d "{
    \"fg_variant_id\": $DRESS_VARIANT_ID,
    \"items\": [
      {
        \"rm_product_id\": $COTTON_FABRIC_ID,
        \"quantity_per_unit\": 2.0
      },
      {
        \"rm_product_id\": $ZIPPER_ID,
        \"quantity_per_unit\": 1.0
      }
    ]
  }" | jq -r '.data.id')

echo "BOM Created: $BOM_ID"
```

### Step 6: Create and Confirm Production Order
```bash
# 10. Create Production Order (Produce 100 dresses)
PROD_ORDER_ID=$(curl -X POST http://localhost:3000/api/production/production-orders \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d "{
    \"fg_variant_id\": $DRESS_VARIANT_ID,
    \"quantity\": 100,
    \"status\": \"DRAFT\"
  }" | jq -r '.data.id')

# 11. Check Raw Material Availability
curl -X GET http://localhost:3000/api/production/production-orders/$PROD_ORDER_ID/check-availability \
  -H "Authorization: Bearer $TOKEN"

# 12. Confirm Production (Consume raw materials, create finished goods with UIDs)
curl -X POST http://localhost:3000/api/production/production-orders/$PROD_ORDER_ID/confirm \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "generateUIDs": true
  }'

echo "Production Order Confirmed: $PROD_ORDER_ID"
```

### Step 7: Create Customer and Sale
```bash
# 13. Create B2B Customer
CUSTOMER_ID=$(curl -X POST http://localhost:3000/api/customers \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Fashion Retailers Ltd",
    "customer_type": "B2B",
    "kra_pin": "P123456789A",
    "credit_limit": 500000.00
  }' | jq -r '.data.id')

# 14. Open Shift (Required for POS sales)
SHIFT_ID=$(curl -X POST http://localhost:3000/api/shifts/open \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "opening_cash": 10000.00
  }' | jq -r '.data.id')

# 15. Create Sale
SALE_ID=$(curl -X POST http://localhost:3000/api/sales \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d "{
    \"sale_type\": \"INVOICE\",
    \"customer_id\": $CUSTOMER_ID,
    \"items\": [
      {
        \"product_id\": $DRESS_ID,
        \"variant_id\": $DRESS_VARIANT_ID,
        \"quantity\": 5,
        \"unit_price\": 2500.00
      }
    ]
  }" | jq -r '.data.id')

echo "Sale Created: $SALE_ID"
```

### Step 8: Process Payment
```bash
# 16. Process Payment
curl -X POST http://localhost:3000/api/payments \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d "{
    \"sale_id\": $SALE_ID,
    \"amount\": 12500.00,
    \"payment_method\": \"CARD\",
    \"reference\": \"CARD-123456789\"
  }"
```

### Step 9: Fiscalise Sale (KRA eTIMS)
```bash
# 17. Fiscalise Sale for KRA eTIMS
curl -X POST http://localhost:3000/api/kra-etims/fiscalise/sale/$SALE_ID \
  -H "Authorization: Bearer $TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "async": false
  }'
```

### Step 10: Generate Reports
```bash
# 18. Get Sales Report
curl -X GET "http://localhost:3000/api/reports/sales/date?start_date=2024-01-01&end_date=2024-12-31" \
  -H "Authorization: Bearer $TOKEN"

# 19. Get Profit/Loss Report
curl -X GET "http://localhost:3000/api/reports/financial/profit-loss?start_date=2024-01-01&end_date=2024-12-31" \
  -H "Authorization: Bearer $TOKEN"

# 20. Get Inventory Report
curl -X GET "http://localhost:3000/api/reports/inventory/low-stock" \
  -H "Authorization: Bearer $TOKEN"
```

---

## Tips for Testing

### 1. Save Token
Always save the JWT token from login response for subsequent requests:
```bash
TOKEN=$(curl -X POST http://localhost:3000/api/auth/login \
  -H "Content-Type: application/json" \
  -d '{"username":"admin","password":"admin123"}' \
  | jq -r '.data.token')
```

### 2. Use Variables
Use shell variables for IDs to avoid hardcoding:
```bash
PRODUCT_ID=$(curl ... | jq -r '.data.id')
echo "Created Product ID: $PRODUCT_ID"
```

### 3. Check Responses
Always check response status codes and error messages:
```bash
curl -X POST ... | jq '.'
```

### 4. Test Error Cases
Test invalid inputs, missing required fields, and unauthorized access:
- Missing required fields
- Invalid data types
- Non-existent IDs
- Unauthorized requests (without token)
- Invalid role access

### 5. Test Complete Workflows
Test complete business workflows:
- Procurement → Production → Sale → Payment → Report
- Sale → Return → Credit Note → Fiscalisation
- Shift Open → Sales → Shift Close

### 6. Use jq for JSON Parsing
Use `jq` for parsing JSON responses in bash scripts:
```bash
# Extract ID
ID=$(curl ... | jq -r '.data.id')

# Pretty print JSON
curl ... | jq '.'

# Extract nested value
VALUE=$(curl ... | jq -r '.data.items[0].price')
```

### 7. Test with Realistic Data
Use clothing industry-specific data:
- Raw materials: fabric, thread, buttons, zippers
- Finished goods: dresses, shirts, pants with size/color variants
- Realistic quantities and prices

### 8. Verify Inventory Changes
After GRN processing, production confirmation, and sales, verify inventory quantities:
```bash
# Check inventory quantity
curl -X GET http://localhost:3000/api/inventory/quantity/$PRODUCT_ID \
  -H "Authorization: Bearer $TOKEN"

# Check inventory movements
curl -X GET "http://localhost:3000/api/inventory/movements?product_id=$PRODUCT_ID" \
  -H "Authorization: Bearer $TOKEN"
```

---

## Common Error Responses

### 401 Unauthorized
**Cause**: Missing or invalid token
```json
{
  "success": false,
  "error": "Unauthorized",
  "message": "Invalid or missing authentication token"
}
```
**Solution**: Include valid JWT token in Authorization header

### 403 Forbidden
**Cause**: Insufficient permissions (wrong role)
```json
{
  "success": false,
  "error": "Forbidden",
  "message": "Insufficient permissions"
}
```
**Solution**: Use user with appropriate role (manager or system_admin)

### 404 Not Found
**Cause**: Resource doesn't exist
```json
{
  "success": false,
  "error": "Not Found",
  "message": "Product with ID 999 not found"
}
```
**Solution**: Verify the resource ID exists

### 400 Bad Request
**Cause**: Invalid input data
```json
{
  "success": false,
  "error": "Bad Request",
  "message": "Invalid request data"
}
```
**Solution**: Check request body format and data types

### 422 Unprocessable Entity
**Cause**: Validation errors
```json
{
  "success": false,
  "error": "Validation Error",
  "message": "Validation failed",
  "errors": [
    {
      "field": "product_type",
      "message": "Product type must be either FG or RM"
    }
  ]
}
```
**Solution**: Fix validation errors listed in response

### 409 Conflict
**Cause**: Resource conflict (e.g., duplicate SKU)
```json
{
  "success": false,
  "error": "Conflict",
  "message": "SKU already exists"
}
```
**Solution**: Use unique SKU or update existing resource

### 500 Internal Server Error
**Cause**: Server-side error
```json
{
  "success": false,
  "error": "Internal Server Error",
  "message": "An unexpected error occurred"
}
```
**Solution**: Check server logs for details

---

## Required Field Reference

### Product Creation
- **Required**: `name`, `product_type` (FG or RM)
- **Optional**: `sku`, `track_inventory` (default: true), `description`, `default_vat_rate`

### Product Variant Creation
- **Required**: `attributes` (JSON object)
- **Optional**: `sku`

### Sale Creation
- **Required**: `items` (array with `product_id`, `quantity`)
- **Optional**: `sale_type` (default: POS), `customer_id`, `items.*.variant_id`, `items.*.unit_price`, `discount_code`

### GRN Creation
- **Required**: `items` (array with `product_id`, `quantity`, `unit_cost`)
- **Optional**: `vendor_id`, `purchase_order_id`, `received_at`, `items.*.variant_id`

### BOM Creation
- **Required**: `fg_variant_id`, `items` (array with `rm_product_id`, `quantity_per_unit`)
- **Optional**: None

### Production Order Creation
- **Required**: `fg_variant_id`, `quantity`
- **Optional**: `status` (default: DRAFT)
- **Note**: BOM must exist for `fg_variant_id` before creating production order

### Return Creation
- **Required**: `sale_id`, `items` (array with `sale_item_id`, `inventory_item_id`, `quantity`)
- **Optional**: `reason`, `items.*.reason`

### Customer Creation
- **Required**: `name`
- **Optional**: `customer_type` (default: B2C), `kra_pin`, `credit_limit`, `price_list_id`

---

## Notes

### Business Context
This POS system is designed for a **clothing manufacturing and retail business** that:
- **Procures** raw materials (fabric, thread, buttons, zippers)
- **Produces** finished goods (dresses, shirts, pants) using BOMs
- **Sells** to both retail (B2C) and wholesale (B2B) customers
- **Tracks** inventory using both quantity-based and UID-based methods
- **Integrates** with KRA eTIMS for fiscal receipt generation

### Important Points

1. **Product Types**:
   - `RM` (Raw Material): Materials used in production (fabric, zippers, etc.)
   - `FG` (Finished Good): Products sold to customers (dresses, shirts, etc.)

2. **Inventory Tracking**:
   - **Quantity-based**: For raw materials and non-UID tracked items
   - **UID-based**: For finished goods that need individual tracking (with barcodes)

3. **Production Workflow**:
   - Create BOM (Bill of Materials) for finished good variant
   - Create production order (specify quantity to produce)
   - Confirm production (consumes raw materials, creates finished goods)

4. **Sales Workflow**:
   - Open shift (required for POS sales)
   - Create sale with items
   - Process payment
   - Fiscalise sale (KRA eTIMS integration)

5. **Returns Workflow**:
   - Create return with sale items
   - Complete return (updates inventory, generates credit note)
   - Fiscalise return (KRA eTIMS credit note)

6. **Authentication**:
   - All endpoints (except `/api/health` and `/api/auth/login`) require JWT token
   - Include token in Authorization header: `Authorization: Bearer YOUR_TOKEN_HERE`

7. **Date Formats**:
   - Use ISO 8601 format: `YYYY-MM-DD` or `YYYY-MM-DDTHH:mm:ssZ`
   - Example: `2024-01-15` or `2024-01-15T10:00:00Z`

8. **Variant Attributes**:
   - Variant attributes must be a JSON object
   - Common attributes for clothing: `size`, `color`
   - Example: `{"size": "M", "color": "Blue"}`

9. **UID Generation**:
   - UIDs are auto-generated when `generateUID: true` is set
   - Barcodes are auto-generated when `generateBarcode: true` is set
   - Used for individual item tracking in finished goods

10. **Transaction Safety**:
    - GRN processing, production confirmation, and return completion use database transactions
    - Ensures data consistency and atomicity

---

## Testing Checklist

Use this checklist to ensure comprehensive API testing:

### Phase 0: User Management
- [ ] Create user (cashier/manager/admin)
- [ ] List users
- [ ] Get user by ID
- [ ] Update user
- [ ] Delete user (soft delete)

### Phase 1: Authentication & Authorization
- [ ] System setup (initial admin user)
- [ ] User login (username/password)
- [ ] Quick POS login (username/PIN)
- [ ] Get current user profile
- [ ] Refresh token (not implemented)
- [ ] Test unauthorized access (missing token)
- [ ] Test forbidden access (wrong role)

### Phase 2: Products
- [ ] Create raw material product (RM)
- [ ] Create finished good product (FG)
- [ ] Create product variant
- [ ] List products
- [ ] Search products
- [ ] Get product by ID
- [ ] Update product
- [ ] Delete product
- [ ] List product variants
- [ ] Update product variant
- [ ] Delete product variant
- [ ] Upload product image

### Phase 3: Inventory (Quantity-Based)
- [ ] Get inventory quantity
- [ ] Check stock availability
- [ ] Adjust inventory quantity
- [ ] Set inventory quantity
- [ ] Set reorder level
- [ ] List inventory
- [ ] Get inventory record
- [ ] Get low stock items
- [ ] Get inventory movements

### Phase 6: Inventory (UID-Based)
- [ ] Create inventory item with UID
- [ ] Get inventory item by UID/barcode
- [ ] Scan inventory item (barcode/UID)
- [ ] Update inventory item status
- [ ] List inventory items

### Sales & POS
- [ ] Create POS sale (walk-in)
- [ ] Create invoice sale (B2B customer)
- [ ] Get sale by ID
- [ ] List sales
- [ ] Cancel sale

### Phase 5: Payments
- [ ] Create payment (cash/bank)
- [ ] Initialize Paystack payment
- [ ] Verify Paystack payment
- [ ] Paystack webhook
- [ ] Process mobile money payment
- [ ] Get payment by ID
- [ ] List payments
- [ ] Get payments for sale

### Phase 7A: Categories
- [ ] Create category
- [ ] List categories
- [ ] Get category hierarchy (tree)
- [ ] Get category by ID
- [ ] Update category
- [ ] Delete category
- [ ] Get product categories
- [ ] Assign product to categories
- [ ] Remove product from category

### Phase 7B: Pricing
- [ ] Create price list
- [ ] Get default price list
- [ ] List price lists
- [ ] Get price list by ID
- [ ] Update price list
- [ ] Delete price list
- [ ] Create product price
- [ ] List product prices
- [ ] Get price (lookup)
- [ ] Get price for customer
- [ ] Get product price by ID
- [ ] Update product price
- [ ] Delete product price

### Phase 7C: Taxes
- [ ] Create tax rate
- [ ] List tax rates
- [ ] Get tax rate by ID
- [ ] Get tax rate by code
- [ ] Update tax rate
- [ ] Delete tax rate
- [ ] Create tax category
- [ ] List tax categories
- [ ] Get tax category by ID
- [ ] Update tax category
- [ ] Delete tax category
- [ ] Create tax exemption
- [ ] List tax exemptions
- [ ] Get tax exemption by ID
- [ ] Update tax exemption
- [ ] Delete tax exemption
- [ ] Calculate product tax
- [ ] Get product tax rate

### Phase 8: Discounts
- [ ] Create discount
- [ ] List discounts
- [ ] Validate discount
- [ ] Apply discount (preview)
- [ ] Get discount by ID
- [ ] Get discount by code
- [ ] Update discount
- [ ] Delete discount
- [ ] Create discount rule
- [ ] List discount rules
- [ ] Get discount rule by ID
- [ ] Update discount rule
- [ ] Delete discount rule

### Phase 9: Procurement
- [ ] Create vendor
- [ ] List vendors
- [ ] Get vendor by ID
- [ ] Update vendor
- [ ] Delete vendor
- [ ] Create purchase order
- [ ] List purchase orders
- [ ] Get purchase order by ID
- [ ] Update purchase order
- [ ] Update purchase order status
- [ ] Delete purchase order
- [ ] Create GRN
- [ ] List GRNs
- [ ] Get GRN by ID
- [ ] Process GRN

### Phase 10: Production
- [ ] Create BOM
- [ ] Get BOM by ID
- [ ] Get BOM by variant ID
- [ ] List BOMs
- [ ] Update BOM
- [ ] Delete BOM
- [ ] Create production order
- [ ] Check raw material availability
- [ ] Confirm production
- [ ] Get production order
- [ ] List production orders

### Returns
- [ ] Create return
- [ ] Complete return
- [ ] Get return
- [ ] List returns

### Phase 11: Returns
- [ ] Create return
- [ ] Complete return
- [ ] Get return
- [ ] List returns

### Phase 12: Shifts & Cash Management
- [ ] Open shift
- [ ] Get current shift
- [ ] List shifts
- [ ] Get shift by ID
- [ ] Close shift
- [ ] Create cash count (mid-shift)
- [ ] List cash counts
- [ ] Get cash count by ID

### Phase 13: KRA eTIMS
- [ ] Create fiscal receipt (manual)
- [ ] List fiscal receipts
- [ ] Get fiscal receipt by ID
- [ ] Fiscalise sale
- [ ] Fiscalise return (credit note)

### Customers
- [ ] Create B2C customer
- [ ] Create B2B customer
- [ ] Get customer
- [ ] Update customer
- [ ] Search customers
- [ ] List customers

### Reports
- [ ] Sales report by date
- [ ] Sales report by product
- [ ] Sales report by customer
- [ ] Sales report by cashier
- [ ] Fast-moving products report
- [ ] Slow-moving products report
- [ ] Low stock items report
- [ ] Profit/loss report
- [ ] Tax report
- [ ] Shift report

### Complete Workflows
- [ ] Procurement → Production → Sale workflow
- [ ] Sale → Return → Credit Note workflow
- [ ] Shift open → Sales → Shift close workflow

---

## Additional Resources

- **Swagger Documentation**: Visit `http://localhost:3000/api-docs` for interactive API documentation
- **Database Models**: See `server/models/` for detailed model structures
- **Validation Rules**: See `server/modules/*/validations/` for detailed validation requirements
- **Service Logic**: See `server/modules/*/services/` for business logic implementation

---

**Last Updated**: Based on codebase review - All models, validations, and business logic verified.
