# Testing Guide

## Overview
This guide provides instructions for writing and running tests for the XYZ POS Server application.

## Test Structure

```
tests/
├── setup.js                    # Test setup file (runs before all tests)
├── unit/                       # Unit tests (isolated service/function tests)
│   ├── services/              # Service unit tests
│   │   ├── auth.test.js
│   │   ├── sales.test.js
│   │   ├── inventory.test.js
│   │   └── ...
│   ├── utils/                 # Utility function tests
│   └── helpers/               # Test helper functions
├── integration/                # Integration tests (API endpoint tests)
│   ├── auth.test.js
│   ├── sales.test.js
│   ├── inventory.test.js
│   └── ...
├── fixtures/                   # Test data fixtures
│   ├── users.js
│   ├── products.js
│   └── ...
└── helpers/                    # Test utilities
    ├── dbHelpers.js           # Database helpers (setup/teardown)
    ├── authHelpers.js         # Authentication helpers
    └── factory.js             # Test data factories
```

## Running Tests

```bash
# Run all tests with coverage
npm test

# Run only unit tests
npm run test:unit

# Run only integration tests
npm run test:integration

# Run tests in watch mode
npm run test:watch

# Run specific test file
npx jest tests/unit/services/auth.test.js

# Run tests matching pattern
npx jest --testNamePattern="should create user"
```

## Test Environment

- **Database**: Uses `xyz_pos_db_test` (separate test database)
- **Environment**: `NODE_ENV=test`
- **Isolation**: Each test should clean up after itself
- **Fixtures**: Use fixtures for consistent test data

## Writing Unit Tests

Unit tests test individual functions/services in isolation.

### Example: Service Unit Test

```javascript
const userService = require('../../modules/users/services');

describe('User Service', () => {
  describe('createUser', () => {
    it('should create a new user', async () => {
      const userData = {
        username: 'testuser',
        password: 'password123',
        full_name: 'Test User',
        role: 'cashier'
      };
      
      const user = await userService.createUser(userData);
      
      expect(user).toHaveProperty('id');
      expect(user.username).toBe(userData.username);
      expect(user.role).toBe('cashier');
    });
    
    it('should throw error for duplicate username', async () => {
      const userData = {
        username: 'duplicate',
        password: 'password123',
        full_name: 'Test User'
      };
      
      await userService.createUser(userData);
      
      await expect(userService.createUser(userData))
        .rejects.toThrow('Username already exists');
    });
  });
});
```

## Writing Integration Tests

Integration tests test API endpoints with the full request/response cycle.

### Example: API Integration Test

```javascript
const request = require('supertest');
const app = require('../../app');
const { setupTestDb, cleanupTestDb } = require('../helpers/dbHelpers');
const { createTestUser, getAuthToken } = require('../helpers/authHelpers');

describe('Sales API', () => {
  let authToken;
  
  beforeAll(async () => {
    await setupTestDb();
    const user = await createTestUser();
    authToken = await getAuthToken(user);
  });
  
  afterAll(async () => {
    await cleanupTestDb();
  });
  
  describe('POST /api/sales', () => {
    it('should create a sale', async () => {
      const saleData = {
        items: [
          { product_id: 1, quantity: 2, unit_price: 100 }
        ]
      };
      
      const response = await request(app)
        .post('/api/sales')
        .set('Authorization', `Bearer ${authToken}`)
        .send(saleData)
        .expect(201);
      
      expect(response.body.data).toHaveProperty('id');
      expect(response.body.data.items).toHaveLength(1);
    });
    
    it('should require authentication', async () => {
      await request(app)
        .post('/api/sales')
        .send({ items: [] })
        .expect(401);
    });
  });
});
```

## Test Data Management

### Using Fixtures

```javascript
const { testUsers, testProducts } = require('../fixtures');

// Use fixture data in tests
const user = await User.create(testUsers.cashier);
const product = await Product.create(testProducts.item1);
```

### Using Factories

```javascript
const factory = require('../helpers/factory');

// Create test data with factory
const user = await factory.user({ role: 'manager' });
const sale = await factory.sale({ 
  items: [
    factory.saleItem({ product_id: 1 })
  ]
});
```

## Database Testing

### Setup/Teardown

```javascript
const { setupTestDb, cleanupTestDb, clearTables } = require('../helpers/dbHelpers');

beforeAll(async () => {
  await setupTestDb(); // Create test database, sync models
});

afterAll(async () => {
  await cleanupTestDb(); // Drop test database
});

beforeEach(async () => {
  await clearTables(); // Clear all tables before each test
});
```

## Testing Best Practices

1. **Isolation**: Each test should be independent
2. **Cleanup**: Always clean up test data
3. **Fixtures**: Use fixtures for consistent test data
4. **Mocks**: Mock external services (Paystack, KRA API)
5. **Coverage**: Aim for >80% code coverage
6. **Naming**: Use descriptive test names
7. **Arrange-Act-Assert**: Structure tests clearly
8. **Error Cases**: Test error scenarios

## Coverage Goals

- **Statements**: >80%
- **Branches**: >80%
- **Functions**: >80%
- **Lines**: >80%

## Performance Testing

```bash
# Run load tests with autocannon
autocannon -c 10 -d 30 http://localhost:3000/api/health

# Run stress tests
artillery quick --count 50 --num 1000 http://localhost:3000/api/health
```

## Security Testing

- Test authentication/authorization
- Test input validation
- Test SQL injection protection
- Test XSS protection
- Test rate limiting

