# KRA eTIMS Integration Setup Guide

## Overview

This POS system includes integration with KRA (Kenya Revenue Authority) eTIMS (Electronic Tax Invoice Management System) for fiscal receipt compliance. This guide explains how to configure and use the integration.

## What is KRA eTIMS?

**KRA eTIMS** is the official tax invoice management system by the Kenya Revenue Authority that:
- Requires businesses to issue fiscal receipts for all sales
- Tracks sales and tax information electronically
- Provides QR codes on receipts for verification
- Ensures compliance with tax regulations

## Current Implementation Status

### ✅ Completed
- Fiscal receipt data model and database structure
- Receipt preparation and formatting logic
- Basic KRA client structure
- API endpoints for fiscal receipt creation
- Integration with sales and returns
- Queue structure (placeholder for background processing)

### ⚠️ Needs Configuration
- **Actual KRA eTIMS API integration** - Currently uses placeholder implementations
- **KRA API credentials** - Need to obtain from KRA
- **Queue system** - Currently uses simple processing (can be upgraded to Bull/BullMQ)

## Configuration

### Environment Variables

Add these to your `server/.env` file:

```env
# KRA eTIMS Configuration
KRA_API_URL=https://etims.kra.go.ke/api        # KRA eTIMS API base URL (check KRA docs for correct URL)
KRA_API_KEY=your_kra_api_key                   # Your KRA API key
KRA_API_SECRET=your_kra_api_secret             # Your KRA API secret
KRA_OSCU_URL=https://your-oscu-url.com         # OSCU (Offline System Control Unit) URL if using offline mode
ENABLE_FISCALISATION=true                      # Enable/disable fiscalisation (set to false for testing)
```

### Getting KRA eTIMS Credentials

1. **Register with KRA eTIMS**
   - Visit KRA eTIMS portal: https://etims.kra.go.ke
   - Register your business/taxpayer account
   - Complete KRA eTIMS registration process

2. **Obtain API Credentials**
   - Apply for API access through KRA eTIMS portal
   - Get your API key and secret
   - Note your business/taxpayer PIN

3. **Configure OSCU (if using offline mode)**
   - Set up Offline System Control Unit
   - Configure OSCU URL
   - Test connectivity

## API Implementation

### Current Structure

The KRA client is located at: `server/modules/kra-etims/services/kraClient.js`

**Key Methods:**
1. `getAuthToken()` - Authenticate with KRA API
2. `fiscaliseSaleReceipt(receiptData)` - Fiscalise a sale receipt
3. `fiscaliseCreditNote(creditNoteData)` - Fiscalise a credit note/return

### Implementation Steps

**Step 1: Update Authentication**

Replace the placeholder in `getAuthToken()`:

```javascript
async getAuthToken() {
  try {
    const response = await this.client.post('/auth/token', {
      api_key: this.apiKey,
      api_secret: this.apiSecret,
      grant_type: 'client_credentials', // Check KRA docs for correct grant type
    });
    
    // Cache token (tokens typically expire after 1 hour)
    this.tokenCache = {
      token: response.data.access_token,
      expiresAt: Date.now() + (response.data.expires_in * 1000),
    };
    
    return this.tokenCache.token;
  } catch (error) {
    logger.error('KRA authentication failed', { error: error.response?.data || error.message });
    throw new Error(`KRA authentication failed: ${error.response?.data?.message || error.message}`);
  }
}
```

**Step 2: Update Fiscalisation Endpoint**

Replace the placeholder in `fiscaliseSaleReceipt()`:

```javascript
async fiscaliseSaleReceipt(receiptData) {
  try {
    if (!this.enabled) {
      return this.getMockResponse(receiptData);
    }

    const token = await this.getAuthToken();
    
    // Format data according to KRA eTIMS API specification
    const requestData = {
      receipt_category: receiptData.receipt_category,
      receipt_type: receiptData.receipt_type,
      customer_pin: receiptData.customer_pin || null,
      business_pin: process.env.KRA_BUSINESS_PIN, // Your business PIN
      items: receiptData.items.map(item => ({
        product_code: item.product_code,
        product_name: item.product_name,
        quantity: item.quantity,
        unit_price: item.unit_price,
        vat_rate: item.vat_rate,
        line_total: item.line_total,
        vat_amount: item.vat_amount,
        // Add other fields as per KRA API specification
      })),
      subtotal: receiptData.subtotal,
      discount_amount: receiptData.discount_amount || 0,
      tax_amount: receiptData.tax_amount,
      total_amount: receiptData.total_amount,
      invoice_number: receiptData.invoice_number,
      receipt_date: receiptData.receipt_date || new Date().toISOString(),
      // Add other required fields per KRA eTIMS API docs
    };

    const response = await this.client.post('/fiscal/receipt', requestData, {
      headers: {
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json',
      },
    });

    return {
      receipt_number: response.data.receipt_number,
      qr_code: response.data.qr_code,
      fiscalised_at: new Date(response.data.fiscalised_at),
      z_number: response.data.z_number, // Z-number from KRA
      fiscal_receipt_number: response.data.fiscal_receipt_number,
    };
  } catch (error) {
    logger.error('KRA fiscalisation failed', {
      error: error.response?.data || error.message,
      receiptData: { invoice_number: receiptData.invoice_number },
    });
    throw new Error(`KRA fiscalisation failed: ${error.response?.data?.message || error.message}`);
  }
}
```

**Step 3: Reference KRA Documentation**

The actual implementation must follow:
- KRA eTIMS API documentation (available from KRA)
- Field requirements and data formats
- Authentication flow
- Error handling codes
- Rate limiting rules

## Queue System

Currently, fiscal receipts are processed synchronously. For production, implement a queue system:

### Option 1: Bull Queue (Recommended)

```bash
npm install bull redis
```

Update `fiscalReceipts.js` to use Bull:

```javascript
const Queue = require('bull');
const fiscalReceiptQueue = new Queue('fiscal-receipts', {
  redis: { host: 'localhost', port: 6379 }
});

// Add to queue
await fiscalReceiptQueue.add('fiscalise', { receiptId: fiscalReceipt.id });

// Process queue
fiscalReceiptQueue.process('fiscalise', async (job) => {
  const { receiptId } = job.data;
  await processFiscalReceipt(receiptId);
});
```

### Option 2: Simple In-Memory Queue (Current)

The current implementation uses simple synchronous processing. This works for low-volume but should be upgraded for production.

## Usage

### Creating Fiscal Receipts

**Via API:**
```bash
POST /api/kra-etims/receipts
{
  "receipt_category": "SALE",
  "receipt_type": "B2C",
  "sale_id": 123,
  "customer_pin": null
}
```

**Automatically (on sale completion):**
The system can automatically create fiscal receipts when sales are completed. Enable this in the sales service.

### Receipt Data Structure

Fiscal receipts include:
- **Receipt Information**: Category (SALE/CREDIT_NOTE), Type (B2C/B2B)
- **Business Information**: Business PIN, business name
- **Customer Information**: Customer PIN (for B2B), customer name
- **Items**: Product details, quantities, prices, VAT rates
- **Totals**: Subtotal, discounts, tax, total amount
- **KRA Data**: Receipt number, QR code, Z-number, fiscal receipt number

## Error Handling

The system handles:
- **API Failures**: Retries with exponential backoff
- **Authentication Errors**: Token refresh and retry
- **Validation Errors**: Detailed error messages
- **Network Timeouts**: Queue for retry

## Testing

### Test Mode

Set `ENABLE_FISCALISATION=false` to use mock responses:

```env
ENABLE_FISCALISATION=false
```

This will:
- Skip actual KRA API calls
- Return mock receipt numbers and QR codes
- Log all fiscalisation attempts
- Allow testing without KRA credentials

### Sandbox Testing

If KRA provides a sandbox environment:
```env
KRA_API_URL=https://sandbox.etims.kra.go.ke/api
```

### Production Testing

1. Start with small transactions
2. Monitor logs for errors
3. Verify receipts in KRA portal
4. Check QR codes are valid

## Monitoring

### Logs

Monitor these log entries:
- `Fiscal receipt created` - Receipt record created
- `Fiscal receipt fiscalised` - Successfully sent to KRA
- `Fiscal receipt failed` - Failed to fiscalise
- `KRA authentication failed` - Auth issues

### Health Checks

Check fiscal receipt status:
```bash
GET /api/kra-etims/receipts?status=PENDING
GET /api/kra-etims/receipts?status=FAILED
```

### Retry Failed Receipts

Failed receipts can be retried:
```bash
POST /api/kra-etims/receipts/:id/retry
```

## Compliance

### Requirements

- ✅ Issue fiscal receipts for all sales
- ✅ Include QR code on receipts
- ✅ Submit to KRA within required timeframe
- ✅ Handle credit notes/returns
- ✅ Maintain audit trail

### Best Practices

1. **Queue Processing**: Use background queue for fiscalisation
2. **Retry Logic**: Implement exponential backoff for failures
3. **Monitoring**: Monitor success/failure rates
4. **Backup**: Store receipts locally before KRA submission
5. **Validation**: Validate data before sending to KRA

## Troubleshooting

### Error: "KRA authentication failed"

**Causes:**
- Invalid API credentials
- API URL incorrect
- Network connectivity issues

**Solutions:**
- Verify API key and secret in `.env`
- Check API URL is correct
- Test network connectivity to KRA servers

### Error: "Fiscal receipt failed"

**Causes:**
- Invalid receipt data format
- Missing required fields
- KRA API validation errors

**Solutions:**
- Check receipt data matches KRA format
- Review KRA API documentation
- Check error details in logs

### Receipts Not Processing

**Causes:**
- Queue not running
- Fiscalisation disabled
- Processing errors

**Solutions:**
- Check queue worker is running
- Verify `ENABLE_FISCALISATION=true`
- Review error logs

## Next Steps

1. **Get KRA Credentials**: Register and obtain API access
2. **Implement API Calls**: Replace placeholders with actual KRA API integration
3. **Test in Sandbox**: Test with KRA sandbox (if available)
4. **Set Up Queue**: Implement Bull queue for production
5. **Monitor**: Set up monitoring and alerts

## Additional Resources

- [KRA eTIMS Portal](https://etims.kra.go.ke)
- [KRA eTIMS API Documentation](https://etims.kra.go.ke/api-docs) (requires KRA account access)
- [KRA Support](https://www.kra.go.ke) - Contact KRA for API access and documentation

---

**Note:** The actual KRA eTIMS API endpoints, authentication methods, and data formats may differ from the examples above. Always refer to the official KRA eTIMS API documentation for accurate implementation details.

