/**
 * Production API endpoints
 */

import apiClient from './client';
import { API_ENDPOINTS } from '@/config/api';

export const productionAPI = {
  /**
   * List BOMs
   * @param {Object} params - Query parameters
   * @returns {Promise}
   */
  listBOMs: async (params = {}) => {
    return apiClient.get(API_ENDPOINTS.PRODUCTION.BOMS, { params });
  },

  /**
   * Get BOM by ID
   * @param {number} id
   * @returns {Promise}
   */
  getBOM: async (id) => {
    return apiClient.get(`${API_ENDPOINTS.PRODUCTION.BOMS}/${id}`);
  },

  /**
   * Get BOM by variant ID
   * @param {number} variantId
   * @returns {Promise}
   */
  getBOMByVariant: async (variantId) => {
    return apiClient.get(`${API_ENDPOINTS.PRODUCTION.BOMS}/variant/${variantId}`);
  },

  /**
   * Create BOM
   * @param {Object} bomData
   * @returns {Promise}
   */
  createBOM: async (bomData) => {
    return apiClient.post(API_ENDPOINTS.PRODUCTION.BOMS, bomData);
  },

  /**
   * Update BOM
   * @param {number} id
   * @param {Object} bomData
   * @returns {Promise}
   */
  updateBOM: async (id, bomData) => {
    return apiClient.put(`${API_ENDPOINTS.PRODUCTION.BOMS}/${id}`, bomData);
  },

  /**
   * Delete BOM
   * @param {number} id
   * @returns {Promise}
   */
  deleteBOM: async (id) => {
    return apiClient.delete(`${API_ENDPOINTS.PRODUCTION.BOMS}/${id}`);
  },

  /**
   * List production orders
   * @param {Object} params - Query parameters
   * @returns {Promise}
   */
  listProductionOrders: async (params = {}) => {
    return apiClient.get(API_ENDPOINTS.PRODUCTION.ORDERS, { params });
  },

  /**
   * Get production order by ID
   * @param {number} id
   * @returns {Promise}
   */
  getProductionOrder: async (id) => {
    return apiClient.get(`${API_ENDPOINTS.PRODUCTION.ORDERS}/${id}`);
  },

  /**
   * Create production order
   * @param {Object} orderData
   * @returns {Promise}
   */
  createProductionOrder: async (orderData) => {
    return apiClient.post(API_ENDPOINTS.PRODUCTION.ORDERS, orderData);
  },

  /**
   * Check raw material availability for production order
   * @param {number} id
   * @returns {Promise}
   */
  checkAvailability: async (id) => {
    return apiClient.get(`${API_ENDPOINTS.PRODUCTION.ORDERS}/${id}/check-availability`);
  },

  /**
   * Confirm production (consume raw materials, create finished goods)
   * @param {number} id
   * @param {Object} options - Options like generateUIDs
   * @returns {Promise}
   */
  confirmProduction: async (id, options = {}) => {
    return apiClient.post(`${API_ENDPOINTS.PRODUCTION.ORDERS}/${id}/confirm`, options);
  },
};

