/**
 * Discounts API endpoints
 */

import apiClient from './client';
import { API_ENDPOINTS } from '@/config/api';

export const discountsAPI = {
  /**
   * List discounts
   * @param {Object} params - Query parameters
   * @returns {Promise}
   */
  list: async (params = {}) => {
    return apiClient.get(API_ENDPOINTS.DISCOUNTS, { params });
  },

  /**
   * Get discount by ID
   * @param {number} id
   * @returns {Promise}
   */
  getById: async (id) => {
    return apiClient.get(`${API_ENDPOINTS.DISCOUNTS}/${id}`);
  },

  /**
   * Get discount by code
   * @param {string} code
   * @returns {Promise}
   */
  getByCode: async (code) => {
    return apiClient.get(`${API_ENDPOINTS.DISCOUNTS}/code/${code}`);
  },

  /**
   * Create discount
   * @param {Object} discountData
   * @returns {Promise}
   */
  create: async (discountData) => {
    return apiClient.post(API_ENDPOINTS.DISCOUNTS, discountData);
  },

  /**
   * Update discount
   * @param {number} id
   * @param {Object} discountData
   * @returns {Promise}
   */
  update: async (id, discountData) => {
    return apiClient.put(`${API_ENDPOINTS.DISCOUNTS}/${id}`, discountData);
  },

  /**
   * Delete discount
   * @param {number} id
   * @returns {Promise}
   */
  delete: async (id) => {
    return apiClient.delete(`${API_ENDPOINTS.DISCOUNTS}/${id}`);
  },

  /**
   * Validate discount
   * @param {Object} validationData
   * @returns {Promise}
   */
  validate: async (validationData) => {
    return apiClient.post(`${API_ENDPOINTS.DISCOUNTS}/validate`, validationData);
  },

  /**
   * Apply discount (preview calculation)
   * @param {Object} applyData - Contains discount_code/discount_id, sale_items, customer_id
   * @returns {Promise}
   */
  apply: async (applyData) => {
    return apiClient.post(`${API_ENDPOINTS.DISCOUNTS}/apply`, applyData);
  },

  /**
   * Get discounts available for a product
   * @param {number} productId
   * @returns {Promise}
   */
  getForProduct: async (productId) => {
    return apiClient.get(`${API_ENDPOINTS.DISCOUNTS}/product/${productId}`);
  },

  /**
   * Discount Rules API
   */
  
  /**
   * List discount rules for a discount
   * @param {number} discountId
   * @returns {Promise}
   */
  listRules: async (discountId) => {
    return apiClient.get(`${API_ENDPOINTS.DISCOUNTS}/${discountId}/rules`);
  },

  /**
   * Create discount rule
   * @param {number} discountId
   * @param {Object} ruleData
   * @returns {Promise}
   */
  createRule: async (discountId, ruleData) => {
    return apiClient.post(`${API_ENDPOINTS.DISCOUNTS}/${discountId}/rules`, ruleData);
  },

  /**
   * Delete discount rule
   * @param {number} discountId
   * @param {number} ruleId
   * @returns {Promise}
   */
  deleteRule: async (discountId, ruleId) => {
    return apiClient.delete(`${API_ENDPOINTS.DISCOUNTS}/${discountId}/rules/${ruleId}`);
  },
};


