'use client';

/**
 * Stock Warning Modal Component
 * Displays stock warnings when adding items with insufficient stock
 */

import { Modal, Button } from '@/components/ui';
import { AlertTriangle, CheckCircle, Package } from 'lucide-react';

export default function StockWarningModal({
  isOpen,
  onClose,
  onProceed,
  warnings = [],
  isLoading = false,
}) {
  if (!isOpen || warnings.length === 0) return null;

  // Count items that need attention
  const partialStock = warnings.filter(w => w.status === 'PARTIAL');
  const outOfStock = warnings.filter(w => w.status === 'OUT_OF_STOCK');

  return (
    <Modal
      isOpen={isOpen}
      onClose={onClose}
      title="Stock Alert"
      size="lg"
    >
      <div className="p-6 space-y-4">
        {/* Warning Icon */}
        <div className="flex items-center gap-3 text-amber-600 dark:text-amber-400">
          <AlertTriangle className="w-6 h-6" />
          <p className="font-semibold text-lg">Some items have insufficient stock</p>
        </div>

        {/* Warning Details */}
        <div className="space-y-3">
          {warnings.map((warning, index) => (
            <div
              key={index}
              className={`
                p-4 rounded-xl border-2
                ${
                  warning.status === 'OUT_OF_STOCK'
                    ? 'bg-red-50 dark:bg-red-900/20 border-red-200 dark:border-red-800'
                    : warning.status === 'PARTIAL'
                    ? 'bg-amber-50 dark:bg-amber-900/20 border-amber-200 dark:border-amber-800'
                    : 'bg-green-50 dark:bg-green-900/20 border-green-200 dark:border-green-800'
                }
              `}
            >
              <div className="flex items-start justify-between">
                <div className="flex-1">
                  <p className="font-semibold text-gray-900 dark:text-white">
                    {warning.product_name}
                  </p>
                  {warning.status === 'OUT_OF_STOCK' && (
                    <p className="text-sm text-red-700 dark:text-red-400 mt-1">
                      ❌ Out of stock (0 available, {warning.requested} requested)
                    </p>
                  )}
                  {warning.status === 'PARTIAL' && (
                    <div className="text-sm text-amber-700 dark:text-amber-400 mt-1 space-y-1">
                      <p>
                        ⚠️ Only <strong>{warning.available}</strong> available,{' '}
                        <strong>{warning.requested}</strong> requested
                      </p>
                      <p className="text-xs">
                        {warning.available} will be sold immediately,{' '}
                        {warning.shortfall} will be preordered
                      </p>
                    </div>
                  )}
                </div>
              </div>
            </div>
          ))}
        </div>

        {/* Info Message */}
        <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-xl p-4">
          <div className="flex items-start gap-3">
            <Package className="w-5 h-5 text-blue-600 dark:text-blue-400 flex-shrink-0 mt-0.5" />
            <div className="text-sm text-blue-700 dark:text-blue-400">
              <p className="font-medium mb-1">How this works:</p>
              <ul className="list-disc list-inside space-y-1 ml-2">
                {partialStock.length > 0 && (
                  <li>Available items will be sold immediately</li>
                )}
                {partialStock.length > 0 && (
                  <li>Shortfall will be added to your preorder queue</li>
                )}
                {outOfStock.length > 0 && (
                  <li>Out of stock items will be preordered entirely</li>
                )}
                <li>You can complete the sale with a deposit</li>
              </ul>
            </div>
          </div>
        </div>

        {/* Actions */}
        <div className="flex gap-3 pt-4">
          <Button
            variant="outline"
            onClick={onClose}
            className="flex-1"
            disabled={isLoading}
          >
            Cancel
          </Button>
          <Button
            variant="primary"
            onClick={onProceed}
            className="flex-1 bg-gradient-to-r from-amber-600 to-orange-600 hover:from-amber-700 hover:to-orange-700"
            disabled={isLoading}
            loading={isLoading}
          >
            Proceed Anyway
          </Button>
        </div>
      </div>
    </Modal>
  );
}

