/**
 * Procurement Page
 * Manage vendors, purchase orders, and GRNs
 */

'use client';

import { useState } from 'react';
import { useRouter } from 'next/navigation';
import { Card, CardHeader, CardTitle, CardContent, Button } from '@/components/ui';
import { ShoppingBag, Package, FileText, ArrowRight } from 'lucide-react';

export default function ProcurementPage() {
  const router = useRouter();

  const procurementModules = [
    {
      id: 'vendors',
      title: 'Vendors',
      description: 'Manage supplier and vendor information',
      icon: ShoppingBag,
      href: '/procurement/vendors',
      color: 'bg-blue-500',
    },
    {
      id: 'purchase-orders',
      title: 'Purchase Orders',
      description: 'Create and manage purchase orders',
      icon: FileText,
      href: '/procurement/purchase-orders',
      color: 'bg-green-500',
    },
    {
      id: 'grns',
      title: 'GRNs (Goods Received Notes)',
      description: 'Process incoming goods and update inventory',
      icon: Package,
      href: '/procurement/grns',
      color: 'bg-purple-500',
    },
  ];

  return (
    <div className="p-6 bg-gray-50 dark:bg-gray-950 space-y-6">
      {/* Header */}
      <div>
        <h1 className="text-3xl font-bold text-gray-900 dark:text-white">Procurement</h1>
        <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
          Manage vendors, purchase orders, and goods received
        </p>
      </div>

      {/* Procurement Modules Grid */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        {procurementModules.map((module) => {
          const Icon = module.icon;
          return (
            <Card
              key={module.id}
              className="hover:shadow-lg transition-shadow cursor-pointer"
              onClick={() => router.push(module.href)}
            >
              <CardContent className="p-6 bg-gray-50 dark:bg-gray-950">
                <div className="flex items-start gap-4">
                  <div className={`w-12 h-12 ${module.color} rounded-lg flex items-center justify-center text-white flex-shrink-0`}>
                    <Icon className="w-6 h-6" />
                  </div>
                  <div className="flex-1">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-1">
                      {module.title}
                    </h3>
                    <p className="text-sm text-gray-600 dark:text-gray-400 mb-4">
                      {module.description}
                    </p>
                    <Button
                      variant="outline"
                      size="sm"
                      className="w-full flex items-center justify-center gap-2"
                      onClick={(e) => {
                        e.stopPropagation();
                        router.push(module.href);
                      }}
                    >
                      View
                      <ArrowRight className="w-4 h-4" />
                    </Button>
                  </div>
                </div>
              </CardContent>
            </Card>
          );
        })}
      </div>
    </div>
  );
}


